const { Server } = require("socket.io");
const express = require("express");
const User = require("../models/userModel.js");

const app = express();

const userSocketMap = {}; // userId -> socketId
const socketUserMap = {}; // socketId -> userId
const activeRooms = {}; // roomName -> { users: Set }

const getReceiverSocketId = (receiverId) => userSocketMap[receiverId];
const getAllOnlineUsers = () => Object.keys(userSocketMap);
const getRoomName = (userId1, userId2) =>
  `room:${[userId1, userId2].sort().join("-")}`;
const isUserOnline = (userId) => !!userSocketMap[userId];

let ioInstance;

// ✅ init socket.io
function initSocketIO(server) {
  ioInstance = new Server(server, {
    cors: {
      origin: "*", // ⚠️ restrict in production
      methods: ["GET", "POST"],
      credentials: true,
    },
    allowEIO3: true, // For Flutter clients
    transports: ["websocket", "polling"],
  });

  ioInstance.on("connection", (socket) => {
    console.group(`🔌 [Socket Connected] ${socket.id}`);
    console.log("Handshake query:", socket.handshake.query);
    console.log("Handshake auth:", socket.handshake.auth);

    const userId = socket.handshake.auth?.userId;

    if (userId && userId !== "undefined") {
      userSocketMap[userId] = socket.id;
      socketUserMap[socket.id] = userId;
      console.log(`✅ User mapped: ${userId} → ${socket.id}`);
    } else {
      console.warn("⚠️ No valid userId provided in handshake.auth");
    }

    // Debug all sockets currently connected
    console.log("📡 Current socket connections:");
    ioInstance.sockets.sockets.forEach((sock, id) => {
      console.log(` - ${id} (userId: ${socketUserMap[id] || "?"})`);
    });
    console.groupEnd();

    // Broadcast online users
    ioInstance.emit("getOnlineUsers", getAllOnlineUsers());
    socket.broadcast.emit("userOnline", { userId });

    // 🔄 Example: handle messageSeen
    socket.on("messageSeen", async ({ senderId, receiverId, seen }) => {
      console.group("👁 messageSeen event");
      console.log({ senderId, receiverId, seen });
      try {
        const senderSocketId = getReceiverSocketId(senderId);
        if (senderSocketId) {
          ioInstance.to(senderSocketId).emit("messageSeen", {
            senderId,
            receiverId,
            seen,
          });
          console.log(`✅ Emitted messageSeen to ${senderSocketId}`);
        } else {
          console.warn(`⚠️ Sender ${senderId} not online`);
        }
      } catch (err) {
        console.error("❌ Error handling messageSeen:", err);
        socket.emit("error", err.message);
      }
      console.groupEnd();
    });
    // 🧩 Join a 1:1 room (sender + receiver share the same stable room name)
    socket.on("joinRoom", ({ senderId, receiverId }, ack) => {
      try {
        // (optional) small guard so only the authenticated socket user can join as senderId
        if (socketUserMap[socket.id] !== senderId) {
          console.warn(
            `⚠️ Socket ${socket.id} tried to join as ${senderId} but is ${
              socketUserMap[socket.id]
            }`
          );
          if (ack) ack({ ok: false, error: "forbidden" });
          return;
        }

        const roomName = getRoomName(senderId, receiverId);

        // init room state
        if (!activeRooms[roomName]) {
          activeRooms[roomName] = { users: new Set() };
        }

        // join socket to room
        socket.join(roomName);
        activeRooms[roomName].users.add(senderId);
        console.log(activeRooms);

        console.log(`🏠 ${senderId} joined room ${roomName}`);
        ioInstance
          .to(roomName)
          .emit("joinRoom", { roomName, userId: senderId });

        if (ack) ack({ ok: true, roomName });
      } catch (err) {
        console.error("❌ joinRoom error:", err);
        if (ack) ack({ ok: false, error: "internal_error" });
      }
    });

    // 🚪 Leave a 1:1 room
    socket.on("leaveRoom", ({ senderId }, ack) => {
      try {
        const userId = socketUserMap[socket.id];
        if (userId !== senderId) {
          console.warn(
            `⚠️ leaveRoom mismatch: socket ${socket.id} is ${userId}, tried to leave as ${senderId}`
          );
          if (ack) ack({ ok: false, error: "forbidden" });
          return;
        }

        for (const roomName in activeRooms) {
          if (activeRooms[roomName].users.has(userId)) {
            // remove from room
            activeRooms[roomName].users.delete(userId);
            socket.leave(roomName);

            ioInstance.to(roomName).emit("userLeftRoom", { roomName, userId });
            console.log(`🚪 ${userId} left room ${roomName}`);

            if (activeRooms[roomName].users.size === 0) {
              delete activeRooms[roomName];
              console.log(`🧹 Room cleaned up: ${roomName}`);
            }
          }
        }

        if (ack) ack({ ok: true });
      } catch (err) {
        console.error("❌ leaveRoom error:", err);
        if (ack) ack({ ok: false, error: "internal_error" });
      }
    });

    // 🔻 disconnect
    socket.on("disconnect", async () => {
      console.group(`❌ [Socket Disconnected] ${socket.id}`);
      const userId = socketUserMap[socket.id];

      if (userId && userSocketMap[userId]) {
        delete userSocketMap[userId];
        delete socketUserMap[socket.id];
        socket.broadcast.emit("userOffline", { userId });

        console.log(`🗑 Removed user ${userId} from maps`);

        // Clear user location from DB
        // try {
        //   await User.findByIdAndUpdate(userId, {
        //     $unset: { placeId: "", longitude: "", latitude: "" },
        //   });
        //   console.log(`📍 Cleared placeId for user ${userId}`);
        // } catch (err) {
        //   console.error(
        //     `❌ Failed to clear placeId for user ${userId}:`,
        //     err.message
        //   );
        // }
      }

      // Handle room cleanup
      for (const roomName in activeRooms) {
        if (activeRooms[roomName].users.has(userId)) {
          activeRooms[roomName].users.delete(userId);
          console.log(`🚪 User ${userId} left room: ${roomName}`);
          ioInstance.to(roomName).emit("userLeftRoom", { roomName, userId });

          if (activeRooms[roomName].users.size === 0) {
            delete activeRooms[roomName];
            console.log(`🧹 Room cleaned up: ${roomName}`);
          }
        }
      }

      ioInstance.emit("getOnlineUsers", getAllOnlineUsers());
      console.groupEnd();
    });
  });
}

// 🔥 Safe helper so controllers can emit by userId
function emitToUser(userId, event, data) {
  const socketId = userSocketMap[userId];
  if (socketId && ioInstance) {
    ioInstance.to(socketId).emit(event, data);
  } else {
    console.warn(`⚠️ Tried to emit to ${userId}, but user is offline`);
  }
}

function getIO() {
  if (!ioInstance) {
    throw new Error("❌ Socket.io not initialized yet!");
  }
  return ioInstance;
}

module.exports = {
  app,
  initSocketIO,
  getIO,
  emitToUser,
  getReceiverSocketId,
  getAllOnlineUsers,
  getRoomName,
  isUserOnline,
  activeRooms,
  userSocketMap,
  socketUserMap,
};
