const axios = require("axios");

// ✅ Valid place types
const validTypes = [
  "lodging",
  "restaurant",
  "bar",
  "airport",
  "hotel",
  "university",
];

// ✅ Distance calculator
function calculateDistance(lat1, lon1, lat2, lon2) {
  const R = 6371e3; // meters
  const toRad = (deg) => (deg * Math.PI) / 180;
  const φ1 = toRad(lat1);
  const φ2 = toRad(lat2);
  const Δφ = toRad(lat2 - lat1);
  const Δλ = toRad(lon2 - lon1);

  const a =
    Math.sin(Δφ / 2) * Math.sin(Δφ / 2) +
    Math.cos(φ1) * Math.cos(φ2) * Math.sin(Δλ / 2) * Math.sin(Δλ / 2);
  const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));

  return R * c; // distance in meters
}

// ✅ Classify venue type
function classifyVenue(place) {
  if (!place || !place.name) {
    return { venueType: "unknown", locationName: "Unknown Location", isValid: false };
  }

  const lowerName = place.name.toLowerCase();

  if (lowerName.includes("co-working") || lowerName.includes("coworking")) {
    return { venueType: "co_working", locationName: place.name, isValid: true };
  }

  if (lowerName.includes("hostel")) {
    return { venueType: "hostel", locationName: place.name, isValid: true }; // use real hostel name
  }

  if (lowerName.includes("student hall") || lowerName.includes("student halls")) {
    return { venueType: "student_hall", locationName: place.name, isValid: true }; // real name
  }

  if (place.types && place.types.includes("university")) {
    return { venueType: "university", locationName: place.name, isValid: true }; // ✅ show actual university name
  }

  // fallback → use first recognized type
  const type = place.types?.find((t) => validTypes.includes(t)) || "unknown";
  return { venueType: type, locationName: place.name, isValid: type !== "unknown" };
}


async function getPlaceDetails(latitude, longitude, fallbackLocation = "Unknown Location") {
  const radius = 50; // meters
  let placeId = null;
  let locationName = fallbackLocation;
  let venueType = "unknown";
  let isValidPlace = false;
  let nearestPlace = null;
  let minDistance = Infinity;

  // Step 1: Nearby Search
  const nearbySearchUrl = `https://maps.googleapis.com/maps/api/place/nearbysearch/json?location=${latitude},${longitude}&radius=${radius}&key=${process.env.Google_API_Key}`;

  try {
    const nearbySearchResponse = await axios.get(nearbySearchUrl);
    const nearbyPlaces = nearbySearchResponse.data.results;
console.log(nearbyPlaces);
    if (nearbyPlaces.length > 0) {
      for (const place of nearbyPlaces) {
        const lowerName = place.name?.toLowerCase() || "";

        // only consider if it matches valid types or special cases
        if (
          place.types.some((type) => validTypes.includes(type)) ||
          lowerName.includes("co-working") ||
          lowerName.includes("coworking") ||
          lowerName.includes("hostel") ||
          lowerName.includes("student hall") ||
          lowerName.includes("student halls")
        ) {
          const distance = calculateDistance(
            latitude,
            longitude,
            place.geometry.location.lat,
            place.geometry.location.lng
          );

          if (distance < minDistance) {
            minDistance = distance;
            nearestPlace = place;
          }
        }
      }

      if (nearestPlace) {
        placeId = nearestPlace.place_id;
        const classified = classifyVenue(nearestPlace);
        venueType = classified.venueType;
        locationName = classified.locationName;
        isValidPlace = classified.isValid;
      } else {
        // fallback: pick first place but mark invalid
        const firstNearbyPlace = nearbyPlaces[0];
        placeId = firstNearbyPlace.place_id;
        const classified = classifyVenue(firstNearbyPlace);
        venueType = classified.venueType;
        locationName = classified.locationName;
        isValidPlace = classified.isValid; // will be false if unknown
      }
    }
  } catch (error) {
    console.warn("⚠️ Google Places API request failed:", error.message);
  }

  // Step 2: Geocoding fallback
  if (!placeId) {
    const geocodeUrl = `https://maps.googleapis.com/maps/api/geocode/json?latlng=${latitude},${longitude}&key=${process.env.Google_API_Key}`;

    try {
      const geocodeResponse = await axios.get(geocodeUrl);
      const geocodeResults = geocodeResponse.data.results;

      if (geocodeResults.length > 0) {
        placeId = geocodeResults[0].place_id;
        locationName = geocodeResults[0].formatted_address;
        venueType = "unknown";
        isValidPlace = false; // always false if geocode fallback
      }
    } catch (error) {
      console.warn("⚠️ Google Geocoding API request failed:", error.message);
    }
  }

  return { placeId, locationName, venueType, isValidPlace, nearestPlace, minDistance };
}

module.exports = getPlaceDetails;
