const Notification = require("../models/notificationModel");
const { emitToUser, getReceiverSocketId } = require("../utils/socket");

/**
 * Create and emit a notification to a user
 */
const createAndEmitNotification = async (userId, message, type, link, senderId) => {
  try {
    // Create and save notification
    const newNotification = new Notification({
      user: userId,
      message,
      type,
      link,
      senderId,
    });
    await newNotification.save();
    console.log("📩 Notification created in DB");

    // If user is online, send real-time notification
    const receiverSocketId = getReceiverSocketId(userId);
    if (receiverSocketId) {
      emitToUser(userId, "New Notification", {
        message: newNotification.message,
        createdAt: newNotification.createdAt,
        link: newNotification.link,
        type: newNotification.type,
        id: newNotification._id,
        senderId,
      });

      // Mark notification as delivered
      newNotification.isPending = false;
      await newNotification.save();

      console.log(`📤 Notification delivered to user ${userId}`);
    } else {
      console.log(`⚠️ User ${userId} is offline, notification saved as pending.`);
    }

    return newNotification;
  } catch (error) {
    console.error("❌ Error creating and emitting notification:", error.message);
    throw error;
  }
};

/**
 * Send all pending notifications to a user when they come online
 */
const sendPendingNotifications = async (userId) => {
  try {
    const notifications = await Notification.find({
      user: userId,
      isPending: true,
    });

    if (notifications.length > 0) {
      notifications.forEach(async (notification) => {
        emitToUser(userId, "New Notification", {
          message: notification.message,
          createdAt: notification.createdAt,
          link: notification.link,
          type: notification.type,
          id: notification._id,
          senderId: notification.senderId,
        });

        notification.isPending = false;
        await notification.save();
      });

      console.log(`📤 Sent ${notifications.length} pending notifications to user ${userId}`);
    } else {
      console.log(`ℹ️ No pending notifications for user ${userId}`);
    }
  } catch (error) {
    console.error("❌ Error sending pending notifications:", error.message);
  }
};

module.exports = {
  createAndEmitNotification,
  sendPendingNotifications,
};
