const path = require("path");
const express = require("express");
const morgan = require("morgan");
const helmet = require("helmet");
const mongoSanitize = require("express-mongo-sanitize");
const xss = require("xss-clean");
const cookieParser = require("cookie-parser");
const bodyParser = require("body-parser");
const compression = require("compression");
const cors = require("cors");
const session = require("express-session");
const passport = require("passport");
const dotenv = require("dotenv");
const cron = require("node-cron");
const mongoose = require("mongoose");
const fs = require("fs");
const https = require("https");

const { app, initSocketIO } = require("./utils/socket");

const User = require("./models/userModel");
const AppError = require("./utils/appError");
const userRouter = require("./routes/userRoutes");
const authRoutes = require("./routes/authRoutes");
const lockerRoutes = require("./routes/lockRoutes");
const appLockRoutes = require("./routes/appLockRoutes");
const privacyRoutes = require("./routes/privacyRoutes");
const messageRoutes = require("./routes/messageRoutes");
const paymentRoutes = require("./routes/paymentRoutes");
const appleRoutes = require("./routes/appleRoutes");
const interestRoutes = require("./routes/interestRoutes");
const passwordResetRoute = require("./routes/passwordResetRoutes");
const locationRoutes = require("./routes/locationRoutes");
const googleRoutes = require("./routes/googleAuthRoutes");
const imgSocketRoutes = require("./routes/imgSocketRoutes");
const linkedinRoutes = require("./routes/linkedinAuthRoutes");
const connectionRoutes = require("./routes/connectionRoutes");
const voiceMessageRoutes = require("./routes/voiceMessageRoutes");
const notificationRoutes = require("./routes/notificationRoutes");
const globalErrorHandler = require("./controllers/errorController");
const pushNotificationRoutes = require("./routes/pushNotificationRoutes");
const profileVisibilityRoutes = require("./routes/profileVisibilityRoutes");
const notificationSettingRoutes = require("./routes/notificationSettingRoutes");

const adminUserRoutes = require("./routes/admin/userRoutes");
const adminMessageRoutes = require("./routes/admin/messageRoutes");
const adminConnectionRoutes = require("./routes/admin/connectionRoutes");
const adminCreditRoutes = require("./routes/admin/creditRoutes");
const adminDashboardRoutes = require("./routes/admin/adminRoutes");
const adminNotificationRoutes = require("./routes/admin/notificationRoutes");

dotenv.config({ path: "./config.env" });

const DB = process.env.DATABASE.replace("<db_password>", process.env.DATABASE_PASSWORD);
mongoose.connect(DB).then(() => console.log("DB Connection Successful!"));

app.set("view engine", "pug");
app.set("views", path.join(__dirname, "views"));

app.use(bodyParser.json());
app.enable("trust proxy");

const corsOptions = {
  origin: process.env.CLIENT_URL || "http://localhost:3000",
  methods: "GET,POST,PUT,DELETE",
  allowedHeaders: "Content-Type,Authorization",
  credentials: true,
};
app.use(cors(corsOptions));
app.use(express.static(path.join(__dirname, "public")));
app.use("/uploads", express.static(path.join(__dirname, "uploads")));
app.use(helmet());
if (process.env.NODE_ENV === "development") app.use(morgan("dev"));
app.use(express.json({ limit: "10kb" }));
app.use(express.urlencoded({ extended: true, limit: "10kb" }));
app.use(cookieParser());
app.use(mongoSanitize());
app.use(xss());
app.use(compression());
app.use(session({
  secret: process.env.SESSION_SECRET || "your-secret-key",
  resave: false,
  saveUninitialized: false,
  cookie: { secure: process.env.NODE_ENV === "production", httpOnly: true, maxAge: 86400000 },
}));
app.use(passport.initialize());
app.use(passport.session());

cron.schedule("* * * * *", async () => {
  try {
    const users = await User.find({ visibilityEndTime: { $lte: Date.now() }, isVisible: false });
    users.forEach(async (user) => {
      user.isVisible = true;
      user.visibilityEndTime = null;
      await user.save();
      console.log(`User ${user._id}'s profile is now visible again.`);
    });
  } catch (err) {
    console.error("Error checking profile visibility:", err);
  }
});

app.use("/api/auth/google", googleRoutes);
app.use("/api/auth/linkedin", linkedinRoutes);
app.use("/api/auth", authRoutes);
app.use("/api/v1/users", userRouter);
app.use("/api/v1/interests", interestRoutes);
app.use("/api/v1/connections", connectionRoutes);
app.use("/api/v1/lock", lockerRoutes);
app.use("/api/messages", messageRoutes);
app.use("/api/v1/img", imgSocketRoutes);
app.use("/api/v1/location", locationRoutes);
app.use("/api/v1/payments", paymentRoutes);
app.use("/api/v1/messages", messageRoutes);
app.use("/api/v1/voice", voiceMessageRoutes);
app.use("/api/v1/notification", notificationRoutes);
app.use("/api/v1/push", pushNotificationRoutes);
app.use("/api/v1/app-lock", appLockRoutes);
app.use("/api/v1/data-privacy", privacyRoutes);
app.use("/api/v1/notification-setting", notificationSettingRoutes);
app.use("/api/v1/profile-visibility", profileVisibilityRoutes);
app.use("/api/v1/password", passwordResetRoute);
app.use("/api/auth/apple", appleRoutes);
app.use("/api/admin/users", adminUserRoutes);
app.use("/api/admin/connections", adminConnectionRoutes);
app.use("/api/admin/messages", adminMessageRoutes);
app.use("/api/admin/credits", adminCreditRoutes);
app.use("/api/admin/dashboard", adminDashboardRoutes);
app.use("/api/admin/notifications", adminNotificationRoutes);

app.get("/", (req, res) => {
  res.status(200).send({ status: true, message: "Welcome to App" });
});
app.get('/favicon.ico', (req, res) => res.status(204));
app.all("*", (req, res, next) => {
  next(new AppError(`Can't find ${req.originalUrl} on this server!`, 404));
});
app.use(globalErrorHandler);

// âœ… HTTPS server + socket init here
const privateKey = fs.readFileSync("path_to_private_key_file.pem", "utf8");
const certificate = fs.readFileSync("path_to_certificate_file.pem", "utf8");
const ca = fs.readFileSync("path_to_ca_certificate_file.pem", "utf8");
const credentials = { key: privateKey, cert: certificate, ca };

const server = https.createServer(credentials, app);
initSocketIO(server);

const port = process.env.PORT || 3030;
server.listen(port, () => {
  console.log(`âœ… HTTPS Server is running on https://api.eladde.store:${port}`);
});

process.on("uncaughtException", (err) => {
  console.error("UNCAUGHT EXCEPTION! ðŸ˜¨ Shutting down...");
  console.error(err.name, err.message);
  process.exit(1);
});

process.on("unhandledRejection", (err) => {
  console.error("UNHANDLED REJECTION! ðŸ”¥ Shutting down...");
  console.error(err.name, err.message);
  server.close(() => process.exit(1));
});

process.on("SIGTERM", () => {
  console.log("ðŸ‘‹ SIGTERM RECEIVED. Shutting down gracefully...");
  mongoose.connection.close(() => {
    console.log("MongoDB connection closed.");
    server.close(() => console.log("ðŸ”¥ Process terminated!"));
  });
});


