const mongoose = require("mongoose");
const validator = require("validator");
const bcrypt = require("bcryptjs");
const admin = require("../utils/firebaseServices"); // Firebase Admin SDK
const NotificationLog = require("./notificationLogModel"); 
const userSchema = new mongoose.Schema(
  {
    firstName: String,
    lastName: String,
    othername: String,
    email: {
      type: String,
      lowercase: true,
      validate: [validator.isEmail, "Please provide a valid email"],
    },
    photo: String,
    loginWith: { type: String, default: 'form' },
    role: { type: String, enum: ["user", "admin"], default: "user" },
    password: { type: String, select: false },
    passwordConfirm: String,
    bio: String,
    location: String,
    placeId: String,
    isValidPlace: { type: Boolean, default: false },
    longitude: Number,
    latitude: Number,
    recentJob: String,
    phone: { type: String, trim: true },
    country: { type: String, trim: true },
    company: String,
    stripeCustomerId: String,
    credits: { type: Number, default: 3 },
    socialLinks: {
      linkedin: { type: String, default: "" },
      github: { type: String, default: "" },
      twitter: { type: String, default: "" },
      facebook: { type: String, default: "" },
      website: { type: String, default: "" },
      instagram: { type: String, default: "" },
      whatsapp: { type: String, default: "" },
      addLink: { type: String, default: "" },
    },
    interests: { type: [String], default: [] },
    otps: [{ otp: String, otpExpires: Date }],
    passwordChangedAt: Date,
    passwordResetExpires: Date,
    active: { type: Boolean, default: false },
    connections: [{ type: mongoose.Schema.Types.ObjectId, ref: "User" }],
    lockTime: { type: Number, default: 0 },
    lastAccessTime: { type: Date, default: Date.now },
    isLocked: { type: Boolean, default: false },
    isVisible: { type: Boolean, default: true },
    visibilityEndTime: { type: Date, default: null },
    totalConnections: { type: Number, default: 0 },
    fcmToken: { type: String, default: null },
    resetToken: String,
    resetTokenExpire: Date,
    activeToken: { type: String, default: null },

    privacyPreferences: {
      allowMessages: { type: Boolean, default: true },
      allowMeetUpRequests: { type: Boolean, default: true },
      seeWhoViewedProfile: { type: Boolean, default: true },
      allowLocationAccess: { type: Boolean, default: true },
      allowMediaAndPhotosAccess: { type: Boolean, default: true },
      allowContactsAccess: { type: Boolean, default: true },
      allowSocialLinksAccess: { type: Boolean, default: true }
    },
    notifications: {
      newMeetUps: { type: Boolean, default: true },
      profileViews: { type: Boolean, default: true },
      messages: { type: Boolean, default: true },
      upcomingEvents: { type: Boolean, default: true },
      appUpdates: { type: Boolean, default: true }
    },
    profileVisibility: {
      hideProfile: { type: Boolean, default: false },
      hideProfileDuration: {
        type: String,
        enum: ["forever", "24_hours", "7_days", "30_days"],
        default: null
      },
      eventMode: { type: Boolean, default: false },
      eventModeAvailability: {
        startTime: { type: String, default: null },
        endTime: { type: String, default: null },
        availableAllDay: { type: Boolean, default: false }
      }
    },
    appLock: {
      enabled: { type: Boolean, default: false },
      lockTiming: {
        type: String,
        enum: ["immediately", "5_minutes", "15_minutes", "45_minutes"],
        default: "immediately"
      },
      biometricsEnabled: { type: Boolean, default: false }
    },
  },
  { timestamps: true }
);
userSchema.index({ email: 1 }, { unique: true }); // already ensured
userSchema.index({ placeId: 1, isValidPlace: 1 }); // for nearby queries
userSchema.index({ interests: 1 }); // for mutual interest queries
userSchema.index({ fcmToken: 1 }); // for notifications
userSchema.index({ active: 1, role: 1 }); // useful for admin dashboards
userSchema.index({ createdAt: -1 }); // for recent users sorting
// Pre-save: password hashing
userSchema.pre("save", async function (next) {
  if (!this.isModified("password")) return next();
  this.password = await bcrypt.hash(this.password, 12);
  this.passwordConfirm = undefined;
  next();
});

// Pre-save: mark passwordChangedAt
userSchema.pre("save", function (next) {
  if (!this.isModified("password") || this.isNew) return next();
  this.passwordChangedAt = Date.now() - 1000;
  next();
});

// Pre-save: detect location change (for .save())
userSchema.pre("save", async function (next) {
  if (!this.isNew) {
    const existing = await this.constructor.findById(this._id).select("placeId latitude longitude");
    if (
      existing &&
      (existing.placeId !== this.placeId ||
        existing.latitude !== this.latitude ||
        existing.longitude !== this.longitude)
    ) {
      this._locationChanged = true;
    }
  }
  next();
});

// Post-save: notify on location change (for .save())
userSchema.post("save", async function (doc) {
  if (!this._locationChanged) return;

  await notifyNearbyUsers(doc);
});

// Post-findOneAndUpdate: for .findByIdAndUpdate() support
userSchema.post("findOneAndUpdate", async function (doc) {
//   console.log("🔥 Hook triggered for findOneAndUpdate");

  if (!doc) return;

  const update = this.getUpdate();
  const set = update.$set || {}; // Support $set usage

  const fieldsChanged = ["placeId", "latitude", "longitude"].some(
    (key) => set[key] !== undefined
  );

//   console.log("📦 Raw update payload:", update);
//   console.log("📍 $set payload:", set);
//   console.log("🔍 Fields changed?", fieldsChanged);

  if (!fieldsChanged) return;

  const fullDoc = await this.model.findById(doc._id).select("placeId interests fcmToken firstName");
//   console.log("📄 Updated user:", fullDoc);

  if (fullDoc) {
    await notifyNearbyUsers(fullDoc);
  }
});


// 🔁 Shared logic for notification
const MAX_NOTIFICATIONS_PER_SESSION = 2;

async function notifyNearbyUsers(user) {
  try {
    if (!user.placeId) return;

    const timeWindow = new Date(Date.now() - 60 * 60 * 1000); // 1 hour
    const notificationsSentCount = await NotificationLog.countDocuments({
      fromUser: user._id,
      placeId: user.placeId,
      createdAt: { $gte: timeWindow }
    });

    if (notificationsSentCount >= MAX_NOTIFICATIONS_PER_SESSION) {
      console.log(`⛔ User ${user._id} already reached max notifications in this venue`);
      return;
    }

    const others = await mongoose.model("User").find({
      placeId: user.placeId,
      _id: { $ne: user._id },
      fcmToken: { $ne: null }
    }).select("fcmToken interests firstName _id");

    let sentCount = notificationsSentCount;

    for (const other of others) {
      if (sentCount >= MAX_NOTIFICATIONS_PER_SESSION) break;

      const shared = other.interests.filter(i => user.interests.includes(i));
      if (shared.length === 0) continue;

      const alreadySent = await NotificationLog.exists({
        fromUser: user._id,
        toUser: other._id,
        placeId: user.placeId,
        createdAt: { $gte: timeWindow }
      });

      const token = typeof other.fcmToken === "string" ? other.fcmToken.trim() : "";

      if (!alreadySent && token && token.length >= 140 && token.length <= 2048) {
        console.log(
          `📤 Notifying ${other._id} about ${user._id} | token: ${token.slice(0, 10)}... len=${token.length}`
        );

        try {
          await admin.messaging().send({
            token,
            notification: {
              title: "New person nearby!",
              body: `${user.firstName || "Someone"} at your location shares your interests.`
            },
            data: {
              type: "mutual_interest_alert",
              otherUserId: user._id.toString(),
              placeId: String(user.placeId)
            }
          });

          await NotificationLog.create({
            fromUser: user._id,
            toUser: other._id,
            placeId: user.placeId
          });

          sentCount++;
        } catch (err) {
          if (err.code === "messaging/registration-token-not-registered") {
            console.warn(`❌ Removing invalid FCM token for user ${other._id}`);
            await mongoose.model("User").updateOne(
              { _id: other._id },
              { $unset: { fcmToken: "" } }
            );
          } else {
            console.error("🔥 Unexpected FCM error:", err);
          }
        }
      } else if (!alreadySent) {
        console.warn(`⚠️ Invalid or missing token for user ${other._id}: "${token}"`);
      }
    }
  } catch (err) {
    console.error("❌ Error in notifyNearbyUsers:", err);
  }
}



// Password validation
userSchema.methods.correctPassword = async function (candidatePassword, userPassword) {
  return await bcrypt.compare(candidatePassword, userPassword);
};

// Profile completion calculator
userSchema.methods.calculateProfileCompletion = function () {
  const fields = [
    "firstName", "lastName", "email", "photo", "bio", "location",
    "recentJob", "company", "phone", "country", "interests"
  ];
  let count = 0;

  fields.forEach((field) => {
    if (field === "interests" && this[field] && this[field].length > 0) {
      count++;
    } else if (this[field] && this[field].toString().trim() !== "") {
      count++;
    }
  });

  return Math.round((count / fields.length) * 100);
};

userSchema.methods.changedPasswordAfter = function (JWTTimestamp) {
  if (this.passwordChangedAt) {
    const changedTimestamp = parseInt(this.passwordChangedAt.getTime() / 1000, 10);
    return JWTTimestamp < changedTimestamp;
  }
  return false;
};

const User = mongoose.model("User", userSchema);
module.exports = User;
