const mongoose = require("mongoose");
const validator = require("validator");
const bcrypt = require("bcryptjs");

const connectionModel = require("./connectionModel"); // Import the connection model

const tempUserSchema = new mongoose.Schema(
  {
    firstName: {
      type: String,
    },
    lastName: {
      type: String,
    },
    otherName: {
      type: String,
    },
    email: {
      type: String,
    },
    photo: {
      type: String,
    },
    role: {
      type: String,
      enum: ["user"],
      default: "user",
    },
    password: {
      type: String,
    },
    passwordConfirm: {
      type: String,
    },
    bio: {
      type: String,
    },
    location: {
      type: String,
    },
    placeId: {
      type: String,
    },
      longitude: {
      type: Number,
    },
      latitude: {
      type: Number,
    },
    recentJob: {
      type: String,
    },
    phone: {
      type: String,
      trim: true,
    },
    country: {
      type: String,
      trim: true,
    },
    company: {
      type: String,
    },
    socialLinks: {
      linkedin: { type: String, default: "" },
      github: { type: String, default: "" },
      twitter: { type: String, default: "" },
      facebook: { type: String, default: "" },
      website: { type: String, default: "" },
      instagram: { type: String, default: "" },
      whatsapp: { type: String, default: "" },
    },
    interests: { type: [String], default: [] },
    otps: [
      {
        otp: {
          type: String,
        },
        otpExpires: {
          type: Date,
        },
      },
    ],
    passwordChangedAt: Date,
    passwordResetExpires: Date,
    active: {
      type: Boolean,
      default: false,
    },
    connections: [
      {
        type: mongoose.Schema.Types.ObjectId,
        ref: "Connection", // Reference to the Connection model
      },
    ],
    lockTime: {
      // New field to store lock time
      type: Number, // Lock time in milliseconds
      default: 0,
    },
    lastAccessTime: {
      // New field to store the last access time
      type: Date,
      default: Date.now,
    },
    isLocked: {
      // To track whether the app is locked
      type: Boolean,
      default: false,
    },
    visible: {
      type: Boolean,
      default: true, // Default value is 'true' (profile visible)
    },
    isVisible: {
      type: Boolean,
      default: true, // Profile is visible by default
    },
    visibilityEndTime: {
      type: Date, // When the profile visibility should end
      default: null,
    },
    totalConnections: {
      type: Number,
      default: 0, // Default to zero connections
    },
  },
  { timestamps: true }
);

// Pre-save hooks, password hashing, etc. (Same as in the original schema)
tempUserSchema.pre("save", async function (next) {
  if (!this.isModified("password")) return next();
  this.password = await bcrypt.hash(this.password, 12);
  this.passwordConfirm = undefined;
  next();
});

tempUserSchema.pre("save", function (next) {
  if (!this.isModified("password") || this.isNew) return next();
  this.passwordChangedAt = Date.now() - 1000;
  next();
});

tempUserSchema.methods.correctPassword = async function (
  candidatePassword,
  userPassword
) {
  return await bcrypt.compare(candidatePassword, userPassword);
};

tempUserSchema.methods.changedPasswordAfter = function (JWTTimestamp) {
  if (this.passwordChangedAt) {
    const changedTimestamp = parseInt(
      this.passwordChangedAt.getTime() / 1000,
      10
    );
    return JWTTimestamp < changedTimestamp;
  }
  return false;
};

const TempUser = mongoose.model("TempUser", tempUserSchema);

module.exports = TempUser;
