const mongoose = require("mongoose");

const notificationSchema = new mongoose.Schema(
  {
    user: {
      type: mongoose.Schema.Types.ObjectId,
      ref: "User",
      required: true,
    },
    message: {
      type: String,
      required: [true, "Notification message is required"],
    },
    type: {
      type: String,
      required: true,
    },
    link: {
      type: String,
    },
    status: {
      type: String,
      enum: ["unread", "read"],
      default: "unread",
    },
    isPending: {
      type: Boolean,
      default: true,
    },
    senderId: {
      type: mongoose.Schema.Types.ObjectId,
      ref: "User",
      required: true,
    },
    deletedAt: {
      type: Date, // Soft delete timestamp
      default: null,
    },
  },
  { timestamps: true }
);

// ✅ **Ensure status is "unread" by default when creating a notification**
notificationSchema.pre("save", function (next) {
  if (!this.status) {
    this.status = "unread";
  }
  next();
});

// ✅ **Mark as read -> Also update isPending**
notificationSchema.pre("updateOne", function (next) {
  if (this._update.status === "read") {
    this._update.isPending = false;
  }
  next();
});

// ✅ **Soft Delete Function**
notificationSchema.methods.softDelete = async function () {
  this.deletedAt = new Date();
  return this.save();
};

// ✅ **Restore Soft Deleted Notification**
notificationSchema.methods.restore = async function () {
  this.deletedAt = null;
  return this.save();
};

// ✅ **Find Only Active (Non-Deleted) Notifications**
notificationSchema.statics.findActive = function (filter = {}) {
  return this.find({ ...filter, deletedAt: null });
};

// ✅ **Auto-Delete Notifications if User or Sender is Deleted**
notificationSchema.pre(/^find/, async function (next) {
  const invalidNotifications = await this.model.aggregate([
    {
      $lookup: {
        from: "users",
        localField: "user",
        foreignField: "_id",
        as: "userData",
      },
    },
    {
      $lookup: {
        from: "users",
        localField: "senderId",
        foreignField: "_id",
        as: "senderData",
      },
    },
    {
      $match: {
        $or: [{ userData: { $size: 0 } }, { senderData: { $size: 0 } }],
      },
    },
  ]);

  if (invalidNotifications.length > 0) {
    const idsToDelete = invalidNotifications.map((n) => n._id);
    await this.model.deleteMany({ _id: { $in: idsToDelete } });
  }

  next();
});

const Notification = mongoose.model("Notification", notificationSchema);

module.exports = Notification;
