const mongoose = require("mongoose");

const messageSchema = new mongoose.Schema(
  {
    senderId: {
      type: mongoose.Schema.Types.ObjectId,
      ref: "User",
      required: true,
    },
    receiverId: {
      type: mongoose.Schema.Types.ObjectId,
      ref: "User",
      required: true,
    },
    text: {
      type: String,
    },
    delivered: {
      type: Boolean,
      default: false,
    },
    images: [
      {
        type: String,
      },
    ],
    seen: {
      type: Boolean,
      default: false,
    },
    audio: {
      type: String,
    },
    read: {
      type: Boolean,
      default: false, // Default value for voice messages
    },
  },
  { timestamps: true }
);

// ✅ **Delete Messages if User is Removed**
messageSchema.pre(/^find/, async function (next) {
  const invalidMessages = await this.model.aggregate([
    {
      $lookup: {
        from: "users",
        localField: "senderId",
        foreignField: "_id",
        as: "senderData",
      },
    },
    {
      $lookup: {
        from: "users",
        localField: "receiverId",
        foreignField: "_id",
        as: "receiverData",
      },
    },
    {
      $match: {
        $or: [{ senderData: { $size: 0 } }, { receiverData: { $size: 0 } }],
      },
    },
  ]);

  if (invalidMessages.length > 0) {
    const idsToDelete = invalidMessages.map((msg) => msg._id);
    await this.model.deleteMany({ _id: { $in: idsToDelete } });
  }

  next();
});

// ✅ **Delete All Messages of a Specific User (When User is Deleted)**
messageSchema.statics.deleteMessagesByUser = async function (userId) {
  await this.deleteMany({ $or: [{ senderId: userId }, { receiverId: userId }] });
};

const Message = mongoose.model("Message", messageSchema);

module.exports = Message;