const mongoose = require("mongoose");

const connectionSchema = new mongoose.Schema(
  {
    sender: {
      type: mongoose.Schema.Types.ObjectId,
      ref: "User",
      required: true,
    },
    receiver: {
      type: mongoose.Schema.Types.ObjectId,
      ref: "User",
      required: true,
    },
    status: {
      type: String,
      enum: ["pending", "accepted", "rejected", ""],
      default: "",
    },
    createdAt: {
      type: Date,
      default: Date.now,
    },
  },
  {
    timestamps: true,
  }
);

// **Middleware: Auto-delete invalid connections before fetching**
connectionSchema.pre(/^find/, async function (next) {
  const invalidConnections = await this.model.aggregate([
    {
      $lookup: {
        from: "users",
        localField: "sender",
        foreignField: "_id",
        as: "senderData",
      },
    },
    {
      $lookup: {
        from: "users",
        localField: "receiver",
        foreignField: "_id",
        as: "receiverData",
      },
    },
    {
      $match: {
        $or: [{ senderData: { $size: 0 } }, { receiverData: { $size: 0 } }], // User not found
      },
    },
  ]);

  if (invalidConnections.length > 0) {
    const idsToDelete = invalidConnections.map((conn) => conn._id);
    await this.model.deleteMany({ _id: { $in: idsToDelete } });
  }

  next();
});


const Connection = mongoose.model("Connection", connectionSchema);

module.exports = Connection;
