const fs = require("fs");
const Message = require("../models/messageModel");
const {
  io,
  getReceiverSocketId,
  getRoomName,
  activeRooms,
  userSocketMap,
  emitToUser,
} = require("../utils/socket.js");
const User = require("../models/userModel");
const admin = require("../utils/firebaseServices");

exports.sendVoiceMessage = async (req, res) => {
  try {
    const senderId = req.user._id;
    const { receiverId } = req.params;

    // Validate input
    if (!senderId || !receiverId) {
      return res.status(400).json({
        status: false,
        error: "Sender and receiver IDs are required",
      });
    }

    if (!req.files || req.files.length === 0) {
      return res.status(400).json({
        status: false,
        error: "No voice message files provided",
      });
    }

    // Limit to 5 voice messages max
    if (req.files.length > 5) {
      return res.status(400).json({
        status: false,
        error: "Maximum 5 voice messages allowed per request",
      });
    }

    // Fetch users in parallel
    const [sender, receiver] = await Promise.all([
      User.findById(senderId).select("firstName lastName photo").lean(),
      User.findById(receiverId)
        .select("fcmToken privacyPreferences blockedUsers")
        .lean(),
    ]);

    // Validate users
    if (!sender || !receiver) {
      return res.status(404).json({
        status: false,
        error: "User not found",
      });
    }

    if (receiver.blockedUsers?.includes(senderId)) {
      return res.status(403).json({
        status: false,
        error: "You have been blocked by this user",
      });
    }

    if (!receiver.privacyPreferences?.allowMessages) {
      return res.status(403).json({
        status: false,
        error: "This user does not allow messages",
      });
    }

    // Check if receiver is active in the room
    const roomName = getRoomName(senderId, receiverId);
    const isReceiverActive = activeRooms[roomName]?.users.has(receiverId);

    // Create messages with proper initial status
    const messages = req.files.map((file) => ({
      senderId,
      receiverId,
      audio: `/uploads/audio/${file.filename}`,
      content: "Voice message",
      seen: isReceiverActive,
      createdAt: new Date(),
    }));

    const savedMessages = await Message.insertMany(messages);

    // Prepare response
    const responseData = {
      status: true,
      message: "Voice messages sent successfully",
      data: savedMessages,
    };

    // Defer real-time updates
    process.nextTick(async () => {
      try {
        const receiverSocketId = getReceiverSocketId(receiverId);

        // Real-time updates if receiver is online
        if (receiverSocketId) {
          // Emit new messages
          savedMessages.forEach((msg) => {
            emitToUser(receiverId, "newMessage", msg);
          });

          // Update conversation list
          const unreadCount = await Message.countDocuments({
            senderId,
            receiverId,
            seen: false,
          });

          const lastMessage = await Message.findOne({
            $or: [
              { senderId, receiverId },
              { senderId: receiverId, receiverId: senderId },
            ],
          })
            .sort({ createdAt: -1 })
            .lean();

          emitToUser(receiverId, "getUnreadMessages", {
            _id: senderId,
            firstName: sender.firstName,
            lastName: sender.lastName,
            photo: sender.photo,
            lastMessage: "Voice message",
            lastMessageDateTime:
              lastMessage?.createdAt.toISOString() || new Date().toISOString(),
            unseenCount: isReceiverActive
              ? unreadCount
              : unreadCount + savedMessages.length,
            online_status: isUserOnline(senderId),
          });
        }

        // Send push notification if receiver is offline
        if (!isReceiverActive && receiver.fcmToken) {
          const notification = {
            notification: {
              title: `${sender.firstName} ${sender.lastName}`,
              body: "🎤 Sent you a voice message",
              image: sender.photo,
            },
            data: {
              type: "new_message",
              senderId: senderId.toString(),
              receiverId: receiverId.toString(),
              messageId: savedMessages[0]._id.toString(),
              click_action: "FLUTTER_NOTIFICATION_CLICK",
              url: `/messages/${senderId}`,
            },
            token: receiver.fcmToken,
          };

          await admin
            .messaging()
            .send(notification)
            .catch((err) => console.error("FCM error:", err));
        }
      } catch (err) {
        console.error("Background processing error:", err);
      }
    });

    res.status(200).json(responseData);
  } catch (err) {
    console.error("Error sending voice messages:", err);

    // Clean up uploaded files if error occurred
    if (req.files) {
      await Promise.all(
        req.files.map((file) =>
          fs
            .unlink(file.path)
            .catch((cleanupErr) =>
              console.error("File cleanup error:", cleanupErr)
            )
        )
      );
    }

    res.status(500).json({
      status: false,
      message: "Failed to send voice messages",
      error: process.env.NODE_ENV === "development" ? err.message : undefined,
    });
  }
};
