const User = require("../models/userModel");
const upload = require("../utils/multerConfig"); // Import the multer config
const axios = require("axios");
const mongoose = require("mongoose");
const path = require("path");
const moment = require("moment");
const Fuzzysort = require("fuzzysort");
const Message = require("../models/messageModel");
const Connection = require("../models/connectionModel");
const DeclinedUser = require("../models/DeclinedUser");
const catchAsync = require("../utils/catchAsync");
const { io, getReceiverSocketId,getRoomName,isUserOnline,activeRooms } = require("../utils/socket.js");
const jwt = require("jsonwebtoken");
const getPlaceDetails = require("../utils/placeHelper");
const {
  getUserCoordinates,
  getLocationName,
} = require("../utils/getUserCoordinates");
let declinedUserIds = [];

const signToken = (id) => {
  return jwt.sign({ id }, process.env.JWT_SECRET, {
    expiresIn: process.env.JWT_EXPIRES_IN,
  });
};

const createSendToken = (user, statusCode, req, res) => {
  const token = signToken(user._id);

  // Ensure the expiration time is a valid number
  const cookieExpireInDays =
    parseInt(process.env.JWT_COOKIE_EXPIRE_IN, 10) || 90; // Default to 90 days if not set

  // Define cookie options
  const cookieOptions = {
    expires: new Date(
      Date.now() + cookieExpireInDays * 24 * 60 * 60 * 1000 // Convert days to milliseconds
    ),
    httpOnly: true, // Prevent client-side script access
    secure: req.secure || req.headers["x-forwarded-proto"] === "https", // Ensure secure cookies in production
    sameSite: "strict", // Helps mitigate CSRF attacks
  };

  // Send cookie
  res.cookie("jwt", token, cookieOptions);

  // Remove sensitive data from the user object
  user.password = undefined;

  // Send response
  res.status(statusCode).json({
    status: true,
    token,
    data: {
      user,
    },
  });
};

exports.updateInterest = catchAsync(async (req, res, next) => {
  const { interests } = req.body;
  const userId = req.user._id; // Get user ID from authenticated token

  // Validate input
  if (!interests || !Array.isArray(interests)) {
    return res.status(400).json({
      status: false,
      message: "Interests must be provided as an array.",
    });
  }

  try {
    // Fetch the current user
    const user = await User.findById(userId);

    if (!user) {
      return res.status(404).json({
        status: false,
        message: "User not found.",
      });
    }

    // Check if the user already has 5 interests
    if (user.interests.length >= 5) {
      return res.status(400).json({
        status: false,
        message: "You can have a maximum of 5 interests.",
      });
    }

    // Determine how many interests can be added without exceeding the limit
    const availableSlots = 5 - user.interests.length;
    const newInterests = interests.slice(0, availableSlots); // Only take the allowed number of interests

    // Update user's interests by appending new ones
    const updatedUser = await User.findByIdAndUpdate(
      userId,
      { $addToSet: { interests: { $each: newInterests } } }, // Append new interests
      { new: true } // Return the updated user document
    );

    res.status(200).json({
      status: true,
      message: "Interests updated successfully.",
      user: updatedUser,
    });
  } catch (error) {
    next(error); // Pass the error to error handling middleware
  }
});

// Define the route to get the current logged-in user's profile
exports.getCurrentUserProfile = async (req, res) => {
  try {
    // Ensure user is authenticated
    const user = req.user;
    if (!user) {
      return res.status(401).json({
        status: false,
        error: "You must be logged in to view this information.",
      });
    }

    // Fetch user's accepted connections
    const acceptedConnections = await Connection.find({
      $or: [{ sender: user._id }, { receiver: user._id }],
      status: "accepted",
    }).select("sender receiver -_id");

    // Extract only user IDs from accepted connections
    const connectedUserIds = acceptedConnections.map((conn) =>
      conn.sender.toString() === user._id.toString() ? conn.receiver : conn.sender
    );

    // Get total count of invites (pending requests where the user is the receiver)
    const totalInvites = await Connection.countDocuments({
      receiver: user._id,
      status: "pending",
    });

    // Calculate profile completion percentage
    const userInstance = new User(user); // Create a model instance to call instance methods
    const completionPercentage = userInstance.calculateProfileCompletion();

    // Update total connections in user object
    user.totalConnections = connectedUserIds.length;

    // Modify the user object to include filtered connections, invite count, and profile completion
    const userProfile = {
      ...user.toObject(),
      connections: connectedUserIds, // Show only accepted connections
      totalInvites, // Show total invites count instead of an array
      completionPercentage, // Include profile completion percentage
    };

    return res.status(200).json({
      status: true,
      data: {
        user: userProfile,
      },
    });
  } catch (error) {
    console.error("Error fetching user profile:", error);
    return res.status(500).json({
      status: false,
      error: "An error occurred while fetching the user profile.",
    });
  }
};




exports.updateInterests = catchAsync(async (req, res, next) => {
  const { interests } = req.body;
  const { id } = req.params;

  // Validate interests length (min 3, max 5)
  if (!interests || !Array.isArray(interests) || interests.length < 3 || interests.length > 5) {
    return res.status(400).json({
      status: false,
      error: "Please select between 3 to 5 interests.",
    });
  }

  // Validate user ID
  if (!mongoose.Types.ObjectId.isValid(id)) {
    return res.status(400).json({
      status: false,
      error: "Invalid user ID format.",
    });
  }

  // Find and update the user in one step, ensuring the updated document is returned
  const updatedUser = await User.findByIdAndUpdate(
    id,
    { interests },
    { new: true, runValidators: true }
  );

  // If user not found, return 404 error
  if (!updatedUser) {
    return res.status(404).json({
      status: false,
      error: "No user found with this ID.",
    });
  }

  // Send token with updated user data
  createSendToken(updatedUser, 200, req, res);
});


exports.updateUserInterests = catchAsync(async (req, res, next) => {
  const { interests } = req.body;
  const userId = req.user._id; // Get user ID from authenticated token

  try {
    // Validate input
     if (!interests || !Array.isArray(interests) || interests.length < 3 || interests.length > 5) {
    return res.status(400).json({
      status: false,
      error: "Please select between 3 to 5 interests.",
    });
  }

    // Update user's interests in the database
    await User.findByIdAndUpdate(userId, { interests });

    res.status(200).json({
      status: true,
      message: "Interests updated successfully.",
    });
  } catch (error) {
    console.error("Error updating interests:", error.message);
    res.status(500).json({
      status: false,
      error: "Something went wrong while updating interests.",
    });
  }
});



exports.updateProfileInfo = catchAsync(async (req, res, next) => {
  const { location, recentJob, company, longitude, latitude } = req.body;
  const { id } = req.params;

  try {
    // ✅ Validate required fields
    if (!recentJob || !company) {
      return res.status(400).json({
        status: false,
        error: "Please provide all required fields (recentJob, company).",
      });
    }

    let placeId = null;
    let locationName = location || "Unknown Location";
    let isValidPlace = false;

    // ✅ If coordinates are provided, resolve place details
    if (latitude && longitude) {
      const details = await getPlaceDetails(latitude, longitude, location);

      placeId = details.placeId;
      locationName = details.locationName;
      isValidPlace = details.isValidPlace;
    }

    // ✅ Check if the user exists
    const user = await User.findById(id);
    if (!user) {
      return res.status(404).json({
        status: false,
        error: "No user found with this ID.",
      });
    }

    // ✅ Update user profile information
    await User.findByIdAndUpdate(id, {
      longitude,
      latitude,
      placeId,
      location: locationName,
      recentJob,
      company,
      isValidPlace,
    });

    // ✅ Send response
    res.status(200).json({
      status: true,
      message:
        "Profile info saved successfully. Proceed to the next step: Interests",
      data: {
        userId: id,
        longitude,
        latitude,
        placeId,
        location: locationName,
        recentJob,
        company,
        isValidPlace,
      },
    });
  } catch (error) {
    console.error("❌ Error updating profile:", error.message);
    res.status(500).json({
      status: false,
      error: "Something went wrong while processing profile info.",
    });
  }
});



// Controller for getting all users
exports.getAllUsers = async (req, res) => {
  try {
    const queryObj = { ...req.query }; // Copy query parameters
    const excludedFields = ["page", "sort", "limit", "fields"];
    excludedFields.forEach((el) => delete queryObj[el]);

    // Advanced filtering
    let queryStr = JSON.stringify(queryObj);
    queryStr = queryStr.replace(/\b(gte|gt|lte|lt)\b/g, (match) => `$${match}`);

    let query = User.find(JSON.parse(queryStr));

    // Sorting
    if (req.query.sort) {
      const sortBy = req.query.sort.split(",").join(" ");
      query = query.sort(sortBy);
    } else {
      query = query.sort("-createdAt"); // Default sorting
    }

    // Field limiting
    if (req.query.fields) {
      const fields = req.query.fields.split(",").join(" ");
      query = query.select(fields);
    } else {
      query = query.select("-__v"); // Exclude __v by default
    }

    // Pagination
    const page = req.query.page * 1 || 1;
    const limit = req.query.limit * 1 || 10;
    const skip = (page - 1) * limit;
    query = query.skip(skip).limit(limit);

    if (req.query.page) {
      const numUsers = await User.countDocuments();
      if (skip >= numUsers) throw new Error("This page does not exist");
    }

    // Execute query
    const users = await query;

    // Send response
    res.status(200).json({
      status: true,
      results: users.length,
      data: {
        users,
      },
    });
  } catch (err) {
    console.error(err);
    res.status(500).json({
      status: false,
      Error: "Something went wrong while fetching users",
    });
  }
};
// Get user by ID
exports.getUserById = async (req, res) => {
  try {
    const userId = req.user?._id; // Ensure userId exists
    const { id } = req.params; // ID of the user to fetch

    if (!userId) {
      return res.status(401).json({
        status: false,
        error: "Unauthorized. User not logged in.",
      });
    }

    // Validate user ID format
    if (!mongoose.Types.ObjectId.isValid(id)) {
      return res.status(400).json({
        status: false,
        error: "Invalid User ID format",
      });
    }

    // Find the user (excluding sensitive fields)
    const user = await User.findById(id)
      .select("-password -otps -__v")
      .lean();

    if (!user) {
      return res.status(404).json({
        status: false,
        error: "User not found",
      });
    }

    // Calculate profile completion percentage
    const userInstance = new User(user); // Create a model instance to access instance methods
    const completionPercentage = userInstance.calculateProfileCompletion();

    // Fetch all accepted connections for the user
    const acceptedConnections = await Connection.find({
      $or: [{ sender: id }, { receiver: id }],
      status: "accepted",
    }).select("sender receiver -_id").lean();

    // Extract connection user IDs
    const connectionIds = acceptedConnections.flatMap((conn) =>
      conn.sender.toString() === id ? conn.receiver.toString() : conn.sender.toString()
    );

    // Get total number of accepted connections
    const totalConnections = connectionIds.length;

    // Find the connection status between the logged-in user and the requested user
    const existingConnection = await Connection.findOne({
      $or: [
        { sender: userId, receiver: id },
        { sender: id, receiver: userId },
      ],
    });

    let requestStatus = "";
    let connectionId = null;

    if (existingConnection) {
      connectionId = existingConnection._id; // Assign connection ID if it exists

      if (existingConnection.status === "accepted") {
        requestStatus = "accepted";
      } else if (existingConnection.status === "pending") {
        requestStatus = existingConnection.sender.toString() === userId.toString() ? "pending" : "requested";
      } else if (existingConnection.status === "rejected") {
        requestStatus = "rejected";
      }
    }

    // Update user's connections & totalConnections in the database
    await User.findByIdAndUpdate(id, {
      connections: connectionIds,
      totalConnections: totalConnections,
    });

    // Send response with updated user details
    res.status(200).json({
      status: true,
      data: {
        user: {
          ...user,
          connections: connectionIds,
          totalConnections: totalConnections,
          completionPercentage, // Include completion percentage
        },
        connectionId, // Send connection ID only if it exists
        connectionStatus: requestStatus,
      },
    });
  } catch (err) {
    console.error("Error fetching user:", err);
    res.status(500).json({
      status: false,
      error: "An error occurred while fetching the user.",
    });
  }
};





exports.getSuggestedUsers = async (req, res) => {
  try {
    const { interests, _id: currentUserId, placeId, isValidPlace } = req.user;
    let { limit = 10 } = req.query;
    limit = parseInt(limit);

    if (!mongoose.Types.ObjectId.isValid(currentUserId)) {
      return res.status(400).json({ status: false, error: "Invalid User ID format" });
    }
    if (!interests || !Array.isArray(interests) || interests.length === 0) {
      return res.status(400).json({ status: false, error: "You have no interests defined in your profile." });
    }
    if (!placeId) {
      return res.status(400).json({ status: false, error: "Place ID is required." });
    }

    // Run connections + declined users in parallel
    const [currentUserConnections, declinedUsers] = await Promise.all([
      Connection.aggregate([
        {
          $match: {
            $or: [{ sender: currentUserId }, { receiver: currentUserId }],
            status: "accepted",
          },
        },
        { $project: { _id: 0, userIds: ["$sender", "$receiver"] } },
        { $unwind: "$userIds" },
        { $group: { _id: null, userIds: { $addToSet: "$userIds" } } },
      ]),
      DeclinedUser.find({ sender: currentUserId }).select("receiver -_id").lean(),
    ]);

    // Build exclusion set (cast to string for consistency)
    const excludedUserIds = new Set([
      ...(currentUserConnections[0]?.userIds || []).map(id => id.toString()),
      currentUserId.toString(),
      ...declinedUsers.map(d => d.receiver.toString()),
    ]);

    const excludedObjectIds = Array.from(excludedUserIds).map(id => new mongoose.Types.ObjectId(id));

    // Run nearUsers + users in parallel
    const [nearUsersRaw, usersRaw] = await Promise.all([
      isValidPlace
        ? User.aggregate([
            {
              $match: {
                _id: { $nin: excludedObjectIds }, // exclude self + connections + declined
                interests: { $in: interests },
                placeId: placeId,
                "profileVisibility.hideProfile": false,
              },
            },
            {
              $addFields: {
                mutualInterestCount: {
                  $size: { $setIntersection: [{ $ifNull: ["$interests", []] }, interests] },
                },
              },
            },
            { $limit: limit },
          ])
        : [],
      User.aggregate([
        {
          $match: {
            _id: { $nin: excludedObjectIds },
            interests: { $in: interests },
          },
        },
        {
          $addFields: {
            mutualInterestCount: {
              $size: { $setIntersection: [{ $ifNull: ["$interests", []] }, interests] },
            },
          },
        },
        { $limit: limit },
      ]),
    ]);

    // Collect suggested user IDs
    const suggestedUserIds = [...nearUsersRaw.map(u => u._id), ...usersRaw.map(u => u._id)];

    // Fetch existing connections in one go
    const existingConnections = await Connection.find({
      $or: [
        { sender: currentUserId, receiver: { $in: suggestedUserIds } },
        { receiver: currentUserId, sender: { $in: suggestedUserIds } },
      ],
    }).lean();

    // Build map with both ID + status
    const connectionMap = {};
    existingConnections.forEach(conn => {
      const otherUserId =
        conn.sender.toString() === currentUserId.toString()
          ? conn.receiver.toString()
          : conn.sender.toString();

      let status = "";
      if (conn.status === "accepted") status = "accepted";
      else if (conn.status === "pending")
        status = conn.sender.toString() === currentUserId.toString() ? "pending" : "requested";
      else if (conn.status === "rejected") status = "rejected";

      connectionMap[otherUserId] = {
        connectionId: conn._id,
        connectionStatus: status,
      };
    });

    // Attach connectionId + connectionStatus to nearUsers
    const nearUsers = nearUsersRaw.map(u => ({
      ...u,
      connectionId: connectionMap[u._id.toString()]?.connectionId || null,
      connectionStatus: connectionMap[u._id.toString()]?.connectionStatus || "",
    }));

    res.status(200).json({
      status: true,
      results: nearUsers.length,
      data: {
        nearUsers: nearUsers.sort((a, b) => b.mutualInterestCount - a.mutualInterestCount),
      },
    });
  } catch (err) {
    console.error("Error fetching suggested users:", err);
    res.status(500).json({
      status: false,
      error: `An error occurred: ${err.message}`,
    });
  }
};

exports.getAllSuggestedUsers = async (req, res) => {
  try {
    const { interests, _id: currentUserId, placeId, isValidPlace } = req.user;
    const { limit } = req.query;

    if (!mongoose.Types.ObjectId.isValid(currentUserId)) {
      return res.status(400).json({ status: false, error: "Invalid User ID format" });
    }

    if (!interests || !Array.isArray(interests) || interests.length === 0) {
      return res.status(400).json({ status: false, error: "You have no interests defined in your profile." });
    }

    if (!placeId) {
      return res.status(400).json({ status: false, error: "Place ID is required." });
    }

    const parsedLimit = limit ? Math.max(1, parseInt(limit)) : null;

    // Fetch user's accepted connections
    const currentUserConnections = await Connection.aggregate([
      {
        $match: {
          $or: [{ sender: currentUserId }, { receiver: currentUserId }],
          status: "accepted",
        },
      },
      {
        $project: {
          _id: 0,
          userIds: ["$sender", "$receiver"],
        },
      },
      {
        $unwind: "$userIds",
      },
      {
        $group: {
          _id: null,
          userIds: { $addToSet: "$userIds" },
        },
      },
    ]);

    const excludedUserIds = currentUserConnections.length > 0
      ? currentUserConnections[0].userIds
      : [];
    excludedUserIds.push(currentUserId);

    // Fetch declined users
    const declinedUsers = await DeclinedUser.find({ sender: currentUserId }).select("receiver -_id");
    const declinedUserIds = declinedUsers.map((declined) => declined.receiver.toString());
    excludedUserIds.push(...declinedUserIds);

    // Initialize as let instead of const
    let allUsers = [];

    if (isValidPlace) {
      allUsers = await User.aggregate([
        {
          $match: {
            _id: { $nin: excludedUserIds },
            placeId: placeId,
            interests: { $exists: true, $ne: [] },
            "profileVisibility.hideProfile": false,
          },
        },
        {
          $addFields: {
            matchedCount: {
              $size: {
                $setIntersection: [
                  { $ifNull: ["$interests", []] },
                  interests,
                ],
              },
            },
          },
        },
        {
          $match: {
            matchedCount: { $gt: 0 },
          },
        },
        {
          $sort: { matchedCount: -1 },
        },
        {
          $limit: parsedLimit || 100,
        },
      ]);
    }

    res.status(200).json({
      status: true,
      results: allUsers.length,
      data: {
        users: allUsers,
      },
    });
  } catch (err) {
    console.error("Error fetching all suggested users:", err);
    res.status(500).json({
      status: false,
      error: `An error occurred: ${err.message}`,
    });
  }
};


exports.declineUser = async (req, res) => {
  try {
    const { userId } = req.body; // User ID to decline
    const { _id: currentUserId, interests } = req.user;

    // Validate the current user ID format
    if (!mongoose.Types.ObjectId.isValid(userId)) {
      return res.status(400).json({
        status: false,
        error: "Invalid User ID format.",
      });
    }

    // Check if the user is already declined
    const alreadyDeclined = await DeclinedUser.findOne({
      sender: currentUserId,
      receiver: userId,
    });

    if (!alreadyDeclined) {
      // Add the declined user to the database
      await DeclinedUser.create({
        sender: currentUserId,
        receiver: userId,
        reason: "User declined",
      });
    }

    // Fetch all declined user IDs
    const declinedUsers = await DeclinedUser.find({ sender: currentUserId }).select("receiver");
    const declinedUserIds = declinedUsers.map((declined) => declined.receiver.toString());

    // Fetch updated users with matching interests, excluding declined and current user
    const users = await User.find({
      _id: { $ne: currentUserId, $nin: declinedUserIds }, // Exclude declined users
      interests: { $in: interests }, // Match users with at least one common interest
    });

    // Sort users based on the number of matching interests
    const sortedUsers = users
      .map((user) => {
        const matchedInterests = user.interests.filter((interest) =>
          interests.includes(interest)
        );
        return {
          ...user.toObject(), // Convert the Mongoose document to a plain object
          matchedCount: matchedInterests.length, // Add the number of matched interests
        };
      })
      .sort((a, b) => b.matchedCount - a.matchedCount); // Sort by the highest match count

    // Take the top 4 users
    const topUsers = sortedUsers.slice(0, 4);

    // Respond with the updated top 4 users
    res.status(200).json({
      status: true,
      results: topUsers.length,
      data: {
        users: topUsers,
      },
    });
  } catch (err) {
    console.error("Error in declining user:", err);
    res.status(500).json({
      status: false,
      error: err,
    });
  }
};

// Controller for updating profile picture
exports.updateProfilePicture = catchAsync(async (req, res, next) => {
  // Handle the upload process
  upload.single("photo")(req, res, async (err) => {
    try {
      if (err) {
        return res.status(400).json({
          status: false,
          error: err.message || "An error occurred while uploading the photo.",
        });
      }

      // Extract user ID from the authenticated request
      const currentUserId = req.user._id;

      // Validate user ID
      if (!mongoose.Types.ObjectId.isValid(currentUserId)) {
        return res.status(400).json({ status: false, error: "Invalid user ID format." });
      }

      // Find user
      const user = await User.findById(currentUserId);
      if (!user) {
        return res.status(404).json({ status: false, error: "User not found." });
      }

      // If no file is uploaded, return an error
      if (req.file) {
        // Normalize file path
      let photoPath = req.file.path.replace(/\\/g, "/");

      // Update user profile picture
      user.photo = photoPath;
      await user.save();

      }
  res.status(200).json({
      status: true,
      users: user,
      
    });
    
    } catch (error) {
      console.error("Profile picture update error:", error);
      res.status(500).json({ status: false, error: "An error occurred while updating the profile picture." });
    }
  });
});

exports.updateProfileBioLink = async (req, res) => {
  try {
    // Check if the request body is empty
    if (!req.body || Object.keys(req.body).length === 0) {
      return res.status(400).json({
        status: false,
        error: "Please provide bio and/or social links to update.",
      });
    }

    const { bio, socialLinks } = req.body;

    // Check if the user exists (assuming req.user.id is populated from authentication middleware)
    const user = await User.findById(req.user.id);
    if (!user) {
      return res.status(404).json({
        status: false,
        error: "User not found.",
      });
    }

    // Update bio if provided
    if (bio !== undefined) {
      user.bio = bio;
    }

    // Update social links if provided
    if (socialLinks) {
      if (socialLinks.linkedin !== undefined) {
        user.socialLinks.linkedin = socialLinks.linkedin || ""; // Set to empty string if blank
      }
      if (socialLinks.github !== undefined) {
        user.socialLinks.github = socialLinks.github || "";
      }
      if (socialLinks.twitter !== undefined) {
        user.socialLinks.twitter = socialLinks.twitter || "";
      }
      if (socialLinks.facebook !== undefined) {
        user.socialLinks.facebook = socialLinks.facebook || "";
      }
      if (socialLinks.website !== undefined) {
        user.socialLinks.website = socialLinks.website || "";
      }
      if (socialLinks.whatsapp !== undefined) {
        user.socialLinks.whatsapp = socialLinks.whatsapp || "";
      }
      if (socialLinks.instagram !== undefined) {
        user.socialLinks.instagram = socialLinks.instagram || "";
      }
      // Check for the addLink field
      if (socialLinks.addLink !== undefined) {
        user.socialLinks.addLink = socialLinks.addLink || ""; // Set to empty string if blank
      }
    }

    // Save the updated user document
    await user.save();

    // Respond with the updated user data
    return res.status(200).json({
      status: true,
      message: "Profile updated successfully!",
      data: {
        bio: user.bio,
        socialLinks: user.socialLinks,
      },
    });
  } catch (error) {
    console.error(error);
    return res.status(500).json({
      status: false,
      message: "An error occurred while updating the profile.",
    });
  }
};


exports.updateCurrentUserProfile = async (req, res) => {
  try {
    // Destructure the incoming request body
    const { firstname, lastname, recentJob, company, phone, country } =
      req.body;

    // Validate required fields: firstname, lastname, and recentJob
    if (!firstname) {
      return res.status(400).json({
        status: false,
        error: "Firstname is required.",
      });
    }
    if (!lastname) {
      return res.status(400).json({
        status: false,
        error: "Lastname is required.",
      });
    }
    if (!recentJob) {
      return res.status(400).json({
        status: false,
        error: "Recent job is required.",
      });
    }

    // Proceed if no errors and if the user is authenticated (this is set by the 'protect' middleware)
    const user = req.user;

    // Update user fields
    user.firstName = firstname;
    user.lastName = lastname;
    user.recentJob = recentJob;
    user.company = company || user.company; // Keep old value if not provided
    user.phone = phone || user.phone; // Keep old value if not provided
    user.country = country || user.country; // Keep old value if not provided

    // If a new photo is uploaded, update the photo path
    if (req.file) {
      const photoPath = path.join("uploads", "photos", req.file.filename);
      // Encode the photo path to be URL safe
      user.photo = `/${photoPath.replace(/\\/g, "/")}`; // Replace backslashes with forward slashes
    }

    console.log(user);

    // Save the updated user information
    await user.save();

    // Return success response with the correctly formatted photo path
    return res.status(200).json({
      status: true,
      message: "User profile updated successfully.",
      data: user,
    });
  } catch (error) {
    console.error(error);
    return res.status(500).json({
      status: false,
      error: "An error occurred while updating the user profile.",
    });
  }
};
// Controller to get the current user's profile completion percentage
exports.getProfileCompletion = async (req, res) => {
  try {
    // `req.user` is set by the `protect` middleware
    const user = req.user;

    // If no user found in request, return an error (this is a fallback, ideally won't happen)
    if (!user) {
      return res.status(404).json({
        status: false,
        message: "User not found",
      });
    }

    // Calculate the profile completion percentage
    const profileCompletion = user.calculateProfileCompletion();

    return res.status(200).json({
      status: true,
      message: "Profile completion fetched successfully",
      data: {
        profileCompletion,
      },
    });
  } catch (error) {
    console.error(error);
    return res.status(500).json({
      status: false,
      message: "An error occurred while calculating the profile completion.",
    });
  }
};
// Route to toggle the user's visibility
exports.toggleVisibility = async (req, res) => {
  try {
    // Assuming you have some kind of authentication to get the current logged-in user
    const user = req.user; // This should be set by some authentication middleware

    // Toggle the 'visible' field
    user.visible = !user.visible;

    // Save the updated user to the database
    await user.save();

    // Send response
    res.status(200).json({
      status: true,
      message: `Profile visibility is now ${
        user.visible ? "visible" : "hidden"
      }`,
    });
  } catch (err) {
    console.error(err);
    res.status(500).json({
      status: false,
      message: "An error occurred while updating visibility",
    });
  }
};

// Controller to set profile visibility
exports.setProfileVisibility = async (req, res) => {
  try {
    const { lockForDuration, unit } = req.body; // lockForDuration (e.g., 1, 2, etc.) and unit (e.g., 'minutes', 'hours', 'days')

    // Validation: Check if lockForDuration and unit are provided
    if (!lockForDuration || !unit) {
      return res.status(400).json({
        status: false,
        error: "lockForDuration and unit are required.",
      });
    }

    // Check if unit is valid
    const validUnits = ["minutes", "hours", "days"];
    if (!validUnits.includes(unit)) {
      return res.status(400).json({
        status: false,
        error: 'Invalid time unit. Please use "minutes", "hours", or "days".',
      });
    }

    // Convert the duration to milliseconds using moment.js
    const durationInMillis = moment
      .duration(lockForDuration, unit)
      .asMilliseconds();
    const visibilityEndTime = new Date(Date.now() + durationInMillis);

    const user = req.user; // Access the current logged-in user

    // Set profile visibility to false and calculate when it should become visible again
    user.isVisible = false;
    user.visibilityEndTime = visibilityEndTime;

    await user.save();

    res.status(200).json({
      status: true,
      message: `Profile visibility is now hidden for ${lockForDuration} ${unit}.`,
      visibilityEndTime,
    });
  } catch (err) {
    console.error(err);
    res.status(500).json({
      status: false,
      error: "Failed to set profile visibility",
    });
  }
};

// Controller to check profile visibility status
exports.checkProfileVisibility = async (req, res) => {
  try {
    const user = req.user; // Access the current logged-in user
    // console.log(user);

    const status = user.isVisible ? "visible" : "hidden";
    res.status(200).json({
      status: true,
      message: `Your profile is currently ${status}`,
      visibilityEndTime: user.visibilityEndTime || "N/A",
    });
  } catch (err) {
    console.error(err);
    res.status(500).json({
      status: false,
      error: "Failed to check profile visibility status",
    });
  }
};

exports.searchUser = async (req, res) => {
  const { query = "", limit = 10 } = req.query;

  if (!query) {
    return res.status(400).json({ message: "Query parameter is required" });
  }

  try {
    // Fetch all users from the database
    const users = await User.find({}, "firstName email").lean();

    // Step 1: Find exact and partial matches using MongoDB query
    const exactMatches = await User.find({
      firstName: { $regex: `^${query}`, $options: "i" },
    })
      .limit(parseInt(limit))
      .lean();

    // Step 2: Use Fuzzysort for fuzzy matching on remaining users
    const fuzzyMatches = Fuzzysort.go(query, users, {
      keys: ["firstName"], // Search against the name field
      threshold: -10000, // Set a threshold to control match tolerance
    });

    // Combine results: exact matches first, followed by fuzzy matches
    const suggestions = [
      ...exactMatches, // Add exact matches
      ...fuzzyMatches
        .slice(0, limit - exactMatches.length) // Limit fuzzy results
        .map((match) => match.obj), // Extract original objects from fuzzy matches
    ].slice(0, limit); // Ensure the total does not exceed the limit

    res.status(200).json({
      status: true,
      User: suggestions,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({ status: false, message: "Server error" });
  }
};


exports.searchMessagedUsers = async (req, res) => {
  try {
    const { query = "" } = req.query; // Search query
    const userId = req.user._id; // Logged-in user's ID

    if (!userId) {
      return res.status(400).json({ error: "User ID is required." });
    }

    // Step 1: Find unique user IDs of people the logged-in user has messaged
    const messages = await Message.aggregate([
      {
        $match: {
          $or: [{ senderId: userId }, { receiverId: userId }],
        },
      },
      { $sort: { createdAt: -1 } }, // Sort messages by newest first
      {
        $group: {
          _id: {
            $cond: {
              if: { $eq: ["$senderId", userId] },
              then: "$receiverId",
              else: "$senderId",
            },
          },
          lastMessage: { $first: "$text" }, // Last message text
          lastMessageAudio: { $first: "$audio" }, // Last message audio (if any)
          lastMessageImages: { $first: "$images" }, // Last message images (if any)
          lastMessageTime: { $first: "$createdAt" }, // Last message timestamp
          unseenCount: {
            $sum: {
              $cond: [
                {
                  $and: [
                    { $eq: ["$receiverId", userId] }, // Message is sent to the logged-in user
                    { $ne: ["$seen", true] }, // Message is not seen
                  ],
                },
                1, // If condition is true, add 1 to the sum
                0, // If condition is false, add 0 to the sum
              ],
            },
          }, // Count of unseen messages
        },
      },
    ]);

    // Extract user IDs and exclude the current logged-in user
    const userIds = messages
      .map((msg) => msg._id)
      .filter((id) => id.toString() !== userId.toString());

    // Step 2: Search and populate user details
    const users = await User.find(
      {
        _id: { $in: userIds }, // Only users in the conversation list
        $or: [
          { firstName: { $regex: query, $options: "i" } },
          { lastName: { $regex: query, $options: "i" } },
          { email: { $regex: query, $options: "i" } },
        ],
      },
      { _id: 1, firstName: 1, lastName: 1, photo: 1 } // Now including the user ID
    ).lean();

    // Step 3: Attach lastMessage, lastMessageTime, unseenCount, and icons to each user
    const userMap = Object.fromEntries(
      messages.map((msg) => [msg._id.toString(), msg]) // Map messages by user ID
    );

    const response = users
      .filter((user) => user._id.toString() !== userId.toString()) // Exclude current logged-in user
      .map((user) => {
        const online_status  = isUserOnline(user._id.toString()); // Check online status
        const lastMessageData = userMap[user._id.toString()];

        // Determine the last message display text based on audio or images
        let lastMessageDisplay = lastMessageData?.lastMessage || null;
        if (lastMessageData?.lastMessageAudio) {
          lastMessageDisplay = "🎤 Voice Message"; // Icon for voice message
        } else if (lastMessageData?.lastMessageImages?.length > 0) {
          lastMessageDisplay = "📷️ Image"; // Icon for image message
        }

      return {
  _id: user._id, // Include user ID
  firstName: user.firstName,
  lastName: user.lastName,
  photo: user.photo,
  lastMessage: lastMessageDisplay, // Updated last message with icons

  lastMessageDateTime: lastMessageData?.lastMessageTime 
    ? new Date(lastMessageData.lastMessageTime).toISOString() // Keep raw date-time for sorting
    : null,



  lastMessageTime: lastMessageData?.lastMessageTime
    ? new Date(lastMessageData.lastMessageTime).toLocaleTimeString("en-GB", {
        hour: "2-digit",
        minute: "2-digit",
      }) // Formats as HH:mm
    : null, // If no message time, set to null

  unseenCount: lastMessageData?.unseenCount || 0, // Total unseen messages
  online_status: online_status , // Add online status
};

      })
      .sort((a, b) => {
  const timeA = a.lastMessageDateTime ? new Date(a.lastMessageDateTime).getTime() : 0;
  const timeB = b.lastMessageDateTime ? new Date(b.lastMessageDateTime).getTime() : 0;
  return timeB - timeA; // Sort by latest message first
});




    res.status(200).json({ status: true, data: response });
  } catch (error) {
    console.error("Error in searchMessagedUsers:", error.message);
    res.status(500).json({ status: false, error: "Internal server error" });
  }
};



// Controller to unhide profile manually
exports.unhideProfile = async (req, res) => {
  try {
    const user = req.user; // Access the current logged-in user
    user.isVisible = true;
    user.visibilityEndTime = null; // Reset the visibility end time
    await user.save();
    res.status(200).json({ status: true, message: "Profile is now visible." });
  } catch (err) {
    console.error(err);
    res.status(500).json({ status: false, error: "Failed to unhide profile" });
  }
};
exports.searchUserInterest = async (req, res) => {
  try {
    const { interestQuery } = req.query; // Search query for the specific interest (e.g., "coding")
    const userId = req.user._id; // Assuming the current user's ID is in req.user._id (after authentication)

    // Check if interestQuery is provided
    if (!interestQuery) {
      return res.status(400).json({ error: "Interest query is required." });
    }

    // Find the current user based on their ID
    const user = await User.findById(userId).select("interests"); // Only select 'interests' field for efficiency

    if (!user) {
      return res.status(404).json({ error: "User not found." });
    }

    // Check if the interestQuery is in the current user's interests array
    if (user.interests.includes(interestQuery)) {
      res.status(200).json({
        status: true,
        message: `User has the interest: ${interestQuery}`,
        interest: interestQuery,
      });
    } else {
      res.status(404).json({
        message: `User does not have the interest: ${interestQuery}`,
      });
    }
  } catch (error) {
    console.error("Error in searchUserInterest:", error.message);
    res.status(500).json({ status: false, error: "Internal server error" });
  }
};

exports.getUserLocation = async (req, res) => {
  try {
    const userId = new mongoose.Types.ObjectId(req.user._id.toString());
    console.log("User ID:", userId);

    if (!mongoose.Types.ObjectId.isValid(userId)) {
      return res.status(400).json({ message: "Invalid User ID" });
    }

    // Step 1: Get user coordinates
    const location = await getUserCoordinates();
    if (!location) {
      return res
        .status(400)
        .json({ message: "Could not get user coordinates" });
    }

    const { latitude, longitude } = location;
    console.log("Coordinates:", latitude, longitude);

    // Step 2: Convert coordinates to address
    const address = await getLocationName(latitude, longitude);

    // Step 3: Update user location in database
    const user = await User.findByIdAndUpdate(
      userId,
      {
        location: {
          type: "Point",
          coordinates: [longitude, latitude], // MongoDB expects [lng, lat]
          address: address || "Unknown Location",
        },
      },
      { new: true, runValidators: true }
    );

    if (!user) {
      return res.status(404).json({ message: "User not found" });
    }

    res.json({
      status: true,
      message: "User location updated successfully!",
      location: user.location,
    });
  } catch (error) {
    console.error("Error updating location:", error.message);
    res
      .status(500)
      .json({ message: "Internal server error", error: error.message });
  }
};

exports.saveFCMToken = async (req, res) => {
  try {
    const userId = req.user._id; // Get the logged-in user's ID
    const { fcmToken } = req.body;

    if (!fcmToken) {
      return res
        .status(400)
        .json({ status: false, message: "FCM Token is required" });
    }

    const user = await User.findByIdAndUpdate(
      userId,
      { fcmToken },
      { new: true }
    );

    return res
      .status(200)
      .json({ status: true, message: "FCM Token saved successfully", user });
  } catch (error) {
    console.error("Error saving FCM Token:", error);
    return res
      .status(500)
      .json({ status: false, message: "Internal Server Error" });
  }
};
exports.getUserCredit = async (req, res) => {
  try {
    const userId = req.user._id; // Get logged-in user ID

    // Fetch user details from MongoDB
    const user = await User.findById(userId).select("credits");

    if (!user) {
      return res.status(404).json({
        status: false,
        message: "User not found",
      });
    }

    res.status(200).json({
      status: true,
      credits: user.credits,
    });
  } catch (error) {
    console.error("Error fetching user credits:", error);
    res.status(500).json({
      status: false,
      message: "An error occurred while fetching user credits.",
    });
  }
};

exports.removeInterest = async (req, res) => {
  try {
    const userId = req.user._id; // Logged-in user ID from middleware
    const { interest } = req.body; // Interest to remove (sent in the request body)

    // Validate input
    if (!interest) {
      return res.status(400).json({
        status: false,
        message: "Interest is required.",
      });
    }

    // Find the user and remove the interest from the interests array
    const updatedUser = await User.findByIdAndUpdate(
      userId,
      { $pull: { interests: interest } }, // Remove the interest
      { new: true } // Return the updated user document
    );

    if (!updatedUser) {
      return res.status(404).json({
        status: false,
        message: "User not found.",
      });
    }

    // Send success response with the updated user
    res.status(200).json({
      status: true,
      message: "Interest removed successfully.",
      user: updatedUser,
    });
  } catch (error) {
    console.error("Error while removing interest:", error);
    res.status(500).json({
      status: false,
      message: "An error occurred while removing the interest.",
    });
  }
};
