const User = require("../models/userModel");
const admin = require("../utils/firebaseServices");

exports.sendPushNotification = async (req, res) => {
  try {
    const { userId, senderId, title, message, link } = req.body; // Added `senderId`

    // Validate required fields
    if (!userId || !senderId || !title || !message) {
      return res.status(400).json({
        status: false,
        message: "userId, senderId, title, and message are required.",
      });
    }

    // Validate userId and senderId format (MongoDB ObjectId)
    if (!userId.match(/^[0-9a-fA-F]{24}$/) || !senderId.match(/^[0-9a-fA-F]{24}$/)) {
      return res.status(400).json({
        status: false,
        message: "Invalid userId or senderId format.",
      });
    }

    // Fetch recipient user & FCM token
    const user = await User.findById(userId);
    if (!user) {
      return res.status(404).json({
        status: false,
        message: "Recipient user not found.",
      });
    }
    if (!user.fcmToken) {
      return res.status(400).json({
        status: false,
        message: "Recipient does not have a registered FCM token.",
      });
    }

    // Prepare push notification payload
    const payload = {
      notification: { title, body: message },
      data: {
        senderId: senderId.toString(), // Convert to string
        receiverId: userId.toString(), // Convert to string
        click_action: "FLUTTER_NOTIFICATION_CLICK", // Handle clicks in Flutter
        url: link || "", // Add link if provided
      },
      token: user.fcmToken,
    };

    // Send push notification
    const response = await admin.messaging().send(payload);

    res.status(200).json({
      status: true,
      message: "Push notification sent successfully!",
      firebaseResponse: response,
    });
  } catch (error) {
    console.error("Error sending notification:", error);

    res.status(500).json({
      status: false,
      message: "Failed to send push notification.",
      error: error.message,
    });
  }
};
