const stripe = require("stripe")(process.env.STRIPE_SECRET_KEY);
const admin = require("firebase-admin"); // Firebase admin for push notifications
const axios = require('axios');
const catchAsync = require("../utils/catchAsync");

const { io, getReceiverSocketId } = require("../utils/socket");
const User = require("../models/userModel");
const Payment = require("../models/paymentModel");
const { createAndEmitNotification } = require("../utils/notificationUtils");

// Apple IAP verification endpoints
const APPLE_IAP_ENDPOINTS = {
  sandbox: 'https://sandbox.itunes.apple.com/verifyReceipt',
  production: 'https://buy.itunes.apple.com/verifyReceipt'
};

// Create Checkout Session
exports.createCheckoutSession = catchAsync(async (req, res, next) => {
  const { credits } = req.body;
  const amount = credits * 0.1867 * 100; // Convert to cents

  const session = await stripe.checkout.sessions.create({
    payment_method_types: ["card"],
    success_url: `${req.protocol}://${req.get(
      "host"
    )}/profile?alert=creditsAdded`,
    cancel_url: `${req.protocol}://${req.get("host")}/buy-credits`,
    customer_email: req.user.email,
    client_reference_id: req.user.id,
    metadata: {
      credits: credits.toString(),
    },
    line_items: [
      {
        price_data: {
          currency: "usd",
          product_data: {
            name: `${credits} Credits Purchase`,
            description: "Add credits to your account",
          },
          unit_amount: amount,
        },
        quantity: 1,
      },
    ],
    mode: "payment",
  });

  res.status(200).json({
    status: "success",
    sessionId: session.id,
  });
});

// Webhook for Checkout Completion
exports.webhookCheckout = (req, res) => {
  const sig = req.headers["stripe-signature"];
  const endpointSecret = process.env.STRIPE_WEBHOOK_SECRET;

  let event;
  try {
    event = stripe.webhooks.constructEvent(req.rawBody, sig, endpointSecret);
  } catch (err) {
    return res.status(400).send(`Webhook Error: ${err.message}`);
  }

  if (event.type === "checkout.session.completed") {
    const session = event.data.object;
    updateUserCredits(session);
  }

  res.status(200).json({ received: true });
};

// Update User Credits and Send Push Notification
const updateUserCredits = async (session) => {
  const userId = session.client_reference_id;
  const credits = 14; // Add 14 credits on successful payment
  const senderId = "System"; // Sender ID can be an admin or system

  const user = await User.findByIdAndUpdate(
    userId,
    { $inc: { credits: credits } }, // Increment credits
    { new: true }
  );

  if (!user) {
    console.error("User not found for credit update.");
    return;
  }

  // Send push notification to user
  if (user.fcmToken) {
    const notificationPayload = {
      notification: {
        title: "Payment Successful ðŸŽ‰",
        body: `Your payment was successful! You've received ${credits} credits.`,
      },
      data: {
        click_action: "FLUTTER_NOTIFICATION_CLICK",
        credits: credits.toString(),
        senderId: senderId, // System or Admin ID
        receiverId: userId, // The user receiving the notification
      },
      token: user.fcmToken,
    };

    try {
      await admin.messaging().send(notificationPayload);
      console.log("Push notification sent successfully!");
    } catch (error) {
      console.error("Error sending push notification:", error);
    }
  }
};

exports.getPaymentDetails = catchAsync(async (req, res, next) => {
  const { paymentId } = req.params;
  const { credits } = req.query; // ✅ Get credits from query string
  const userId = req.user.id;

  console.log(`📥 Received request to get payment details for payment ID: ${paymentId} with credits: ${credits}`);

  try {
    // 1. Find user
    const user = await User.findById(userId).select('+stripeCustomerId');
    if (!user) {
      console.log("❌ User not found");
      return res.status(404).json({
        status: "error",
        message: "User not found"
      });
    }

    // 2. Handle Stripe customer
    let stripeCustomer;
    try {
      stripeCustomer = await handleStripeCustomer(user);
    } catch (stripeError) {
      console.log("❌ Stripe customer handling failed:", stripeError.message);
      return res.status(500).json({
        status: "error",
        message: "Payment processing failed - customer issue"
      });
    }

    // 3. Retrieve payment intent
    let paymentIntent;
    try {
      paymentIntent = await stripe.paymentIntents.retrieve(paymentId);
      if (!paymentIntent) {
        throw new Error("Payment intent not found");
      }
    } catch (error) {
      console.log("❌ Payment intent retrieval failed:", error.message);
      return res.status(404).json({
        status: "error",
        message: "Payment not found or invalid"
      });
    }

    // 4. Save payment record
    const creditsAwarded = parseInt(credits) || 0;

    const paymentData = {
      paymentId: paymentIntent.id,
      amount: paymentIntent.amount,
      currency: paymentIntent.currency,
      status: paymentIntent.status,
      created: new Date(paymentIntent.created * 1000),
      customerId: stripeCustomer.id,
      user_id: user._id,
      credits: creditsAwarded,
    };

    await Payment.create(paymentData);

    // 5. Handle successful payment
    if (paymentIntent.status === "succeeded" && creditsAwarded > 0) {
      await User.findByIdAndUpdate(
        userId,
        { $inc: { credits: creditsAwarded } },
        { new: true }
      );

      sendPaymentSuccessNotification(user, paymentData, creditsAwarded)
        .catch(err => console.error("⚠️ Notification error (ignored):", err.message));
    }

    // 6. Final response
    return res.status(200).json({
      status: true,
      payment: paymentData,
      creditsAwarded,
    });

  } catch (error) {
    console.error("❌ Unexpected error in payment processing:", error);
    throw error;
  }
});

// Helper functions remain the same...


// Helper function to handle Stripe customer
async function handleStripeCustomer(user) {
  if (user.stripeCustomerId) {
    try {
      const customer = await stripe.customers.retrieve(user.stripeCustomerId);
      if (!customer.deleted) return customer;
    } catch (err) {
      console.log("⚠️ Existing Stripe customer invalid, creating new...");
    }
  }

  // Create new customer
  const newCustomer = await stripe.customers.create({
    email: user.email,
    name: `${user.firstName} ${user.lastName}`,
    metadata: { userId: user._id.toString() }
  });

  user.stripeCustomerId = newCustomer.id;
  await user.save();

  return newCustomer;
}

// Helper function to send payment notifications
async function sendPaymentSuccessNotification(user, payment, creditsAwarded) {
  const notificationMessage = `Payment successful! ${creditsAwarded} credits added to your account.`;
  const notificationLink = `/payments/${payment.paymentId}`;

  try {
    // Socket notification
    await createAndEmitNotification(
      io,
      user._id,
      notificationMessage,
      "Payment Success",
      notificationLink,
      user._id // Sender is the system in this case
    );

    // Firebase push notification
    if (user.fcmToken && user.notifications?.paymentSuccess) {
      const fcmPayload = {
        notification: {
          title: "Payment Processed",
          body: notificationMessage,
        },
        data: {
          type: "payment_success",
          paymentId: payment.paymentId,
          creditsAwarded: creditsAwarded.toString(),
          click_action: "FLUTTER_NOTIFICATION_CLICK",
          url: notificationLink,
        },
        token: user.fcmToken,
      };

      await admin.messaging().send(fcmPayload)
        .catch(err => console.error("FCM error:", err));
    }
  } catch (err) {
    console.error("Notification error:", err);
  }
}

/**
 * Process Apple In-App Purchase (direct transaction without receipt verification)
 * POST /api/payments/verify-iap
 * 
 * @param {string} productId - The product ID being purchased
 */
exports.verifyIAPReceipt = catchAsync(async (req, res, next) => {
  const { productId, purchaseId } = req.body;
  const userId = req.user.id;

  if (!productId) {
    return res.status(400).json({
      status: "error",
      message: "Product ID is required"
    });
  }

  try {
    // Check if this purchase has already been processed
    const existingPurchase = await Payment.findOne({ 'iapData.transactionId': purchaseId });
    if (existingPurchase) {
      return res.status(200).json({
        status: 'success',
        message: 'This purchase has already been processed',
        alreadyProcessed: true
      });
    }

    // Find the user
    const user = await User.findById(userId);
    if (!user) {
      return res.status(404).json({
        status: "error",
        message: "User not found"
      });
    }

    // Map product IDs to credit amounts
    const productToCredits = {
      'com.eladde.app.fiveCredits': 5,
      'com.eladde.app.mainCredits': 10,
      'com.eladde.app.threeCredits': 3,
      // Add more product IDs as needed
    };

    // Map product IDs to amounts (for record keeping)
    const productToAmount = {
      'com.eladde.app.fiveCredits': 2.99,
      'com.eladde.app.mainCredits': 3.99,
      'com.eladde.app.threeCredits': 1.99,
      // Add more product IDs as needed
    };

    const credits = productToCredits[productId];
    const amount = productToAmount[productId];
    if (!credits) {
      return res.status(400).json({
        status: "error",
        message: "Unrecognized product ID"
      });
    }


    const transactionId = purchaseId;
    // Create a new payment record
    const payment = await Payment.create({
      paymentId: transactionId,
      amount: amount,
      currency: 'USD',
      status: 'completed',
      user_id: userId,
      paymentMethod: 'apple_iap',
      credits: credits,
      iapData: {
        productId,
        transactionId,
        processedAt: new Date()
      }
    });

    // Update user's credits
    user.credits = (user.credits || 0) + credits;
    await user.save({ validateBeforeSave: false });

    // Send notification
    await sendPaymentSuccessNotification(user, payment, credits);

    res.status(200).json({
      status: 'success',
      data: {
        creditsAdded: credits,
        newBalance: user.credits,
        purchaseId: payment.paymentId
      }
    });

  } catch (error) {
    console.error('IAP Processing Error:', error);
    return res.status(500).json({
      status: "error",
      message: "Failed to process IAP transaction"
    });
  }
});

// Retrieve User Payments
exports.getUserPayments = catchAsync(async (req, res, next) => {
  try {
    const user = await User.findById(req.user.id);
    if (!user) {
      return res.status(404).json({ status: "error", message: "User not found" });
    }

    const payments = await Payment.find({ user_id: user._id });

    if (payments.length === 0) {
      return res.status(404).json({ status: "error", message: "No payments found" });
    }

    res.status(200).json({
      status: true,
      payments: payments,
    });
  } catch (error) {
    res.status(500).json({ status: false, message: error.message });
  }
});
