const User = require("../models/userModel");
const Email = require("../utils/email");
const bcrypt = require("bcryptjs");

exports.sendResetCode = async (req, res) => {
  const { email } = req.body;

  try {
    const user = await User.findOne({ email });
    if (!user) {
      return res.status(404).json({ status: false, message: "User not found" });
    }

    const otp = Math.floor(10000 + Math.random() * 90000).toString(); // 5-digit OTP
    const otpExpiry = Date.now() + 10 * 60 * 1000; // 10 minutes

    user.resetToken = otp;
    user.resetTokenExpire = otpExpiry;
    await user.save();

    const emailSender = new Email(user, { OTP: otp });
    await emailSender.sendPasswordReset();

    res.status(200).json({
      status: true,
      message: "OTP sent to your email successfully",
      ...(process.env.NODE_ENV === "development" && { code: otp }),
    });
  } catch (error) {
    console.error("sendResetCode error:", error);
    res.status(500).json({ status: false, error: "Internal server error" });
  }
};

exports.verifyResetCode = async (req, res) => {
  const { email, code } = req.body;

  try {
    const user = await User.findOne({ email, resetToken: code });

    if (!user || user.resetTokenExpire < Date.now()) {
      return res.status(400).json({ status: false, message: "Invalid or expired OTP" });
    }

    res.status(200).json({ status: true, message: "OTP verified successfully" });
  } catch (error) {
    console.error("verifyResetCode error:", error);
    res.status(500).json({ status: false, error: "Internal server error" });
  }
};

exports.resetPassword = async (req, res) => {
  const { email, code, newPassword } = req.body;

  try {
    const user = await User.findOne({ email, resetToken: code });

    if (!user || user.resetTokenExpire < Date.now()) {
      return res.status(400).json({ status: false, message: "Invalid or expired OTP" });
    }

    user.password = await bcrypt.hash(newPassword, 10);
    user.resetToken = undefined;
    user.resetTokenExpire = undefined;
    await user.save();

    res.status(200).json({ status: true, message: "Password has been reset successfully" });
  } catch (error) {
    console.error("resetPassword error:", error);
    res.status(500).json({ status: false, error: "Internal server error" });
  }
};
