const Notification = require("../models/notificationModel"); // Notification model
const moment = require("moment");

/**
 * Get notifications for the logged-in user
 */

exports.getNotifications = async (req, res) => {
  try {
    const userId = req.user._id;

    const notifications = await Notification.find({ user: userId, deletedAt: null })
      .populate("senderId", "firstName lastName photo")
      .sort({ createdAt: -1 });

    if (notifications.length === 0) {
      return res.status(200).json({
        status: true,
        message: "No notifications available.",
        notifications: [],
      });
    }

    const formattedNotifications = notifications.map((notification) => {
      const createdAt = moment(notification.createdAt);
      let formattedTime;

      if (createdAt.isSame(moment(), "day")) {
        formattedTime = `${createdAt.format("HH:mm")}`;
      } else if (createdAt.isSame(moment().subtract(1, "day"), "day")) {
        formattedTime = `Yesterday`;
      } else if (createdAt.isSameOrAfter(moment().subtract(7, "days"), "day")) {
        formattedTime = `${createdAt.format("dddd")}`; // Example: "Monday, 14:30"
      } else {
        formattedTime = createdAt.format("DD MMMM YYYY"); // Example: "12 March 2024, 14:30"
      }

      return {
        ...notification.toObject(),
        formattedTime,
      };
    });

    res.status(200).json({
      status: true,
      message: "Notifications fetched successfully.",
      notifications: formattedNotifications,
    });
  } catch (error) {
    console.error("Error fetching notifications:", error.message);
    res.status(500).json({ status: false, message: "An error occurred while fetching notifications." });
  }
};



/**
 * Mark a notification as read
 */
exports.markAsRead = async (req, res) => {
  try {
    const { notificationId } = req.params;
    const notification = await Notification.findById(notificationId);

    if (!notification) {
      return res.status(404).json({ status: false, message: "Notification not found." });
    }

    notification.status = "read";
    notification.isPending = false;
    await notification.save();

    res.status(200).json({ status: true, message: "Notification marked as read." });
  } catch (error) {
    console.error("Error marking notification as read:", error.message);
    res.status(500).json({ status: false, message: "An error occurred while marking notification as read." });
  }
};

/**
 * Mark all notifications as read for the logged-in user
 */
exports.markAllAsRead = async (req, res) => {
  try {
    const userId = req.user._id;

    const result = await Notification.updateMany(
      { user: userId, status: "unread", deletedAt: null },
      { $set: { status: "read", isPending: false } }
    );

    res.status(200).json({
      status: true,
      message: "All notifications marked as read.",
      modifiedCount: result.modifiedCount, // Number of updated notifications
    });
  } catch (error) {
    console.error("Error marking all notifications as read:", error.message);
    res.status(500).json({ status: false, message: "An error occurred while marking notifications as read." });
  }
};


/**
 * Soft delete a notification
 */
exports.softDeleteNotification = async (req, res) => {
  try {
    const { notificationId } = req.params;
    const notification = await Notification.findById(notificationId);

    if (!notification) {
      return res.status(404).json({ status: false, message: "Notification not found." });
    }

    notification.deletedAt = new Date(); // Set the soft delete timestamp
    await notification.save();

    res.status(200).json({ status: true, message: "Notification deleted successfully." });
  } catch (error) {
    console.error("Error deleting notification:", error.message);
    res.status(500).json({ status: false, message: "An error occurred while deleting notification." });
  }
};

exports.softDeleteAllNotifications = async (req, res) => {
  try {
    const userId = req.user._id; // Get user ID from authentication middleware

    // Mark all notifications as deleted by setting `deletedAt` timestamp
    const result = await Notification.updateMany(
      { user: userId, deletedAt: null }, // Only update non-deleted notifications
      { $set: { deletedAt: new Date() } }
    );

    res.status(200).json({
      status: true,
      message: "All notifications soft deleted successfully.",
      updatedCount: result.modifiedCount, // Number of notifications updated
    });
  } catch (error) {
    console.error("Error soft deleting notifications:", error.message);
    res.status(500).json({
      status: false,
      error: "Something went wrong while soft deleting notifications.",
    });
  }
};
