const User = require("../models/userModel.js");
const Message = require("../models/messageModel.js");
const {
  io,
  getReceiverSocketId,
  getRoomName,
  activeRooms,
  userSocketMap,
  emitToUser,
  isUserOnline,
} = require("../utils/socket.js");
const catchAsync = require("../utils/catchAsync.js");
const mongoose = require("mongoose");
const moment = require("moment");
const { createAndEmitNotification } = require("../utils/notificationUtils.js");
const admin = require("../utils/firebaseServices");
exports.getUsersForSidebar = async (req, res) => {
  try {
    const loggedInUserId = req.user._id;
    console.log(loggedInUserId);
    const filteredUsers = await User.find({
      _id: { $ne: loggedInUserId },
    }).select("-password");

    res.status(200).json(filteredUsers);
  } catch (error) {
    console.error("Error in getUsersForSidebar: ", error.message);
    res.status(500).json({ error: "Internal server error" });
  }
};

/**
 * Get chat messages between the logged-in user and a specific user.
 */
exports.getMessages = async (req, res) => {
  try {
    const { id: userToChatId } = req.params;
    const myId = req.user._id;
    const { page = 1, limit = 20 } = req.query;

    if (!mongoose.Types.ObjectId.isValid(userToChatId)) {
      return res
        .status(400)
        .json({ status: false, message: "Invalid user ID" });
    }

    const userToChatObjectId = new mongoose.Types.ObjectId(userToChatId);

    // ✅ Fetch full user details
    const userToChat = await User.findById(userToChatObjectId).lean();

    if (!userToChat) {
      return res.status(404).json({ status: false, message: "User not found" });
    }

    // ✅ Add online status
    userToChat.online_status = !!getReceiverSocketId(userToChatObjectId);

    const skip = (parseInt(page) - 1) * parseInt(limit);

    // ✅ Fetch Messages with pagination
    const messages = await Message.find({
      $or: [
        { senderId: myId, receiverId: userToChatObjectId },
        { senderId: userToChatObjectId, receiverId: myId },
      ],
    })
      .sort({ createdAt: -1 })
      .skip(skip)
      .limit(parseInt(limit))
      .select("senderId receiverId text images audio createdAt seen") // keep this limited for performance
      .populate("senderId", "firstName lastName photo _id")
      .populate("receiverId", "firstName lastName photo _id")
      .lean();

    // ✅ Format timestamps
    messages.forEach((message) => {
      message.formattedTime = moment(message.createdAt).format("HH:mm");
    });

    // ✅ Count Total Messages
    const totalMessages = await Message.countDocuments({
      $or: [
        { senderId: myId, receiverId: userToChatObjectId },
        { senderId: userToChatObjectId, receiverId: myId },
      ],
    });

    res.status(200).json({
      status: true,
      user: userToChat, // now includes all user fields
      messages,
      pagination: {
        totalMessages,
        currentPage: parseInt(page),
        totalPages: Math.ceil(totalMessages / parseInt(limit)),
      },
    });
  } catch (error) {
    console.error("Error in getMessages:", error);
    res.status(500).json({ status: false, error: "Internal server error" });
  }
};


exports.getLastMessages = async (req, res) => {
  try {
    const { id: userToChatId } = req.params; // ID of the user to chat with
    const myId = req.user._id; // Get the logged-in user's ID from JWT

    // Validate userToChatId
    if (!mongoose.Types.ObjectId.isValid(userToChatId)) {
      return res
        .status(400)
        .json({ status: false, message: "Invalid user ID" });
    }

    const userToChatObjectId = new mongoose.Types.ObjectId(userToChatId);

    // Fetch messages between the logged-in user and the other user
    const messages = await Message.find({
      $or: [
        { senderId: myId, receiverId: userToChatObjectId },
        { senderId: userToChatObjectId, receiverId: myId },
      ],
    })
      .sort({ createdAt: 1 }) // Sort by time (oldest first)
      .populate(
        "senderId",
        "firstName lastName email  recentJob company photo seen"
      )
      .populate(
        "receiverId",
        "firstName lastName email  recentJob company photo"
      );

    // Get the last message between users
    const lastMessage = await Message.findOne({
      $or: [
        { senderId: myId, receiverId: userToChatObjectId },
        { senderId: userToChatObjectId, receiverId: myId },
      ],
    })
      .sort({ createdAt: -1 }) // Get the most recent message
      .populate("senderId", "firstName lastName email")
      .populate("receiverId", "firstName lastName email");

    // If no messages exist
    if (!messages.length) {
      return res.status(404).json({
        status: false,
        message: "No messages found",
      });
    }

    // Respond with messages and the last message
    res.status(200).json({
      status: true,
      lastMessage, // Adding the last message
    });
  } catch (error) {
    console.error("Error in getMessages:", error);
    res.status(500).json({ status: false, error: "Internal server error" });
  }
};
/**
 * Send a message to a specific user, emit it via Socket.IO, and create a notification.
 */
exports.sendMessage = async (req, res) => {
  try {
    const { text } = req.body;
    const { id: receiverId } = req.params;
    const senderId = req.user?._id || req.cookies.guestId;

    // Validate input
    if (!text || !text.trim()) {
      return res.status(400).json({
        status: false,
        error: "Message text is required",
      });
    }

    if (!senderId || !receiverId) {
      return res.status(400).json({
        status: false,
        error: "Sender and receiver are required",
      });
    }

    // Fetch users
    const [receiver, sender] = await Promise.all([
      User.findById(receiverId)
        .select("fcmToken privacyPreferences notifications blockedUsers")
        .lean(),
      User.findById(senderId).select("firstName lastName photo").lean(),
    ]);

    // Check user existence and permissions
    if (!receiver || !sender) {
      return res.status(404).json({
        status: false,
        error: "User not found",
      });
    }

    if (receiver.blockedUsers?.includes(senderId)) {
      return res.status(403).json({
        status: false,
        error: "You have been blocked by this user",
      });
    }

    if (!receiver.privacyPreferences?.allowMessages) {
      return res.status(403).json({
        status: false,
        error: "This user does not allow messages",
      });
    }

    // Check if receiver is in the room (synchronously)
    const roomName = getRoomName(receiverId, senderId);
    const isReceiverActive = activeRooms[roomName]?.users.has(receiverId);

    // Create message with proper initial status
    const newMessage = new Message({
      senderId,
      receiverId,
      text: text.trim(),
      status: isReceiverActive ? "seen" : "sent",
      seen: isReceiverActive,
      ...(isReceiverActive && { seenAt: new Date() }),
    });

    await newMessage.save();

    // Update sender's message count (fire-and-forget)
    User.updateOne({ _id: senderId }, { $inc: { messageCount: 1 } }).catch(
      console.error
    );

    // Prepare response data
    const responseData = {
      status: true,
      message: "Message sent successfully",
      data: newMessage.toObject(),
    };

    // Defer real-time updates
    process.nextTick(async () => {
      try {
        // ✅ safe
        emitToUser(receiverId, "newMessage", newMessage);

        // Update conversation list
        const unreadCount = await Message.countDocuments({
          senderId,
          receiverId,
          seen: false,
        });

        emitToUser(receiverId, "getUnreadMessages", {
          _id: senderId,
          firstName: sender.firstName,
          lastName: sender.lastName,
          photo: sender.photo,
          lastMessage: text,
          lastMessageDateTime: newMessage.createdAt.toISOString(),
          unseenCount: isReceiverActive ? unreadCount : unreadCount + 1,
          online_status: isUserOnline(senderId),
        });

        // Send push notification if receiver is offline
        if (!isReceiverActive && receiver.fcmToken) {
          const notification = {
            notification: {
              title: `${sender.firstName} ${sender.lastName}`,
              body: text.length > 50 ? `${text.substring(0, 47)}...` : text,
              image: sender.photo,
            },
            data: {
              type: "new_message",
              senderId: senderId.toString(),
              receiverId: receiverId.toString(),
              messageId: newMessage._id.toString(),
              click_action: "FLUTTER_NOTIFICATION_CLICK",
            },
            token: receiver.fcmToken,
          };

          await admin
            .messaging()
            .send(notification)
            .catch((err) => console.error("FCM error:", err));
        }
      } catch (err) {
        console.error("Background processing error:", err);
      }
    });

    // Send response with correct status
    res.status(201).json(responseData);
  } catch (error) {
    console.error("Error in sendMessage:", error);
    res.status(500).json({
      status: false,
      error: "Internal server error",
      ...(process.env.NODE_ENV === "development" && { details: error.message }),
    });
  }
};

exports.getMessagedUsers = async (req, res) => {
  try {
    if (!req.user || !req.user._id) {
      return res.status(401).json({ status: false, message: "Unauthorized" });
    }

    const userId = new mongoose.Types.ObjectId(req.user._id);

    // Find distinct users the logged-in user has messaged
    const messages = await Message.find({
      $or: [{ senderId: userId }, { receiverId: userId }],
    }).select("senderId receiverId text createdAt");

    const userIds = new Set();
    const lastMessages = {};

    messages.forEach((msg) => {
      const otherUserId =
        msg.senderId.toString() === userId.toString()
          ? msg.receiverId?.toString()
          : msg.senderId?.toString();

      if (otherUserId) {
        userIds.add(otherUserId);

        // Keep track of the last message for each user
        if (
          !lastMessages[otherUserId] ||
          new Date(msg.createdAt) >
            new Date(lastMessages[otherUserId].createdAt)
        ) {
          lastMessages[otherUserId] = {
            text: msg.text,
            createdAt: msg.createdAt,
          };
        }
      }
    });

    // Fetch user details
    const users = await User.find({ _id: { $in: [...userIds] } }).select(
      "firstName lastName photo"
    );

    // Attach last message to each user
    const usersWithLastMessage = users.map((user) => ({
      _id: user._id,
      firstName: user.firstName,
      lastName: user.lastName,
      photo: user.photo,
      lastMessage: lastMessages[user._id.toString()] || null,
    }));

    return res.status(200).json({
      status: true,
      data: usersWithLastMessage,
    });
  } catch (error) {
    console.error("Error in getMessagedUsers:", error);
    return res
      .status(500)
      .json({ status: false, message: "Internal server error" });
  }
};
// Handle marking a message as seen
exports.seenMessage = async (req, res) => {
  try {
    const receiverId = req.user._id; // Get authenticated user's ID
    const { senderId } = req.body; // Get sender ID from request body
    if (!req.body || !req.body.senderId) {
      return res
        .status(400)
        .json({ status: false, error: "Sender ID is required" });
    }
    if (!senderId) {
      return res.status(400).json({
        status: false,
        error: "Sender ID is required.",
      });
    }
    const seen = true;
    // Update only unseen messages where the user is the receiver and sender matches
    const result = await Message.updateMany(
      { receiverId, senderId, seen: false },
      { $set: { seen: true } }
    );
    console.log(senderId);
    emitToUser(senderId, "messagesSeen", { senderId, receiverId, seen });
    return res.status(200).json({
      status: true,
      message: `Messages from sender ${senderId} marked as seen`,
      data: {
        modifiedCount: result.modifiedCount, // Number of messages updated
      },
    });
  } catch (error) {
    console.error("Error updating message seen status:", error);
    return res.status(500).json({
      status: false,
      error: "Error updating message seen status",
    });
  }
};

// Handle updating a message
exports.updateMessage = async (req, res) => {
  try {
    const { messageId, newText, userId } = req.body;
    const message = await Message.findById(messageId);

    if (!message) return res.status(404).json({ error: "Message not found" });

    // Only allow sender to update the message
    if (message.senderId.toString() !== userId) {
      return res
        .status(403)
        .json({ error: "Not authorized to update this message" });
    }

    message.text = newText;
    await message.save();

    res.status(200).json({ status: true, messageId, newText });
  } catch (error) {
    console.error("Error updating message:", error);
    res.status(500).json({ error: "Error updating message" });
  }
};

// Handle deleting a message
exports.deleteMessage = async (req, res) => {
  try {
    const { messageId, userId } = req.body;
    const message = await Message.findById(messageId);

    if (!message) return res.status(404).json({ error: "Message not found" });

    // Only allow sender to delete the message
    if (message.senderId.toString() !== userId) {
      return res
        .status(403)
        .json({ error: "Not authorized to delete this message" });
    }

    await Message.findByIdAndDelete(messageId);

    res
      .status(200)
      .json({ status: true, message: "message delete successfully" });
  } catch (error) {
    console.error("Error deleting message:", error);
    res.status(500).json({ error: "Error deleting message" });
  }
};
// Handle typing status
exports.typingStatus = (req, res) => {
  try {
    const { userId, receiverId, typing } = req.body;

    emitToUser(receiverId, "typing", { userId, typing });

    res.status(200).json({ status: true, typingStatus: typing });
  } catch (error) {
    console.error("Error handling typing status:", error);
    res.status(500).json({ error: "Error handling typing status" });
  }
};
exports.updateMessage = async (req, res) => {
  try {
    const { messageId } = req.params;
    const { newText } = req.body;
    const userId = req.user._id; // Use logged-in user

    // Find message
    const message = await Message.findById(messageId);
    if (!message) {
      return res
        .status(404)
        .json({ status: false, error: "Message not found" });
    }

    // Only sender can edit
    if (message.senderId.toString() !== userId.toString()) {
      return res.status(403).json({ status: false, error: "Not authorized" });
    }

    // Update text
    message.text = newText;
    message.edited = true; // Optional flag to indicate edit
    await message.save();

    // Emit event to receiver via Socket.IO

    emitToUser(receiverId, "messageUpdated", {
      messageId,
      newText,
    });

    res
      .status(200)
      .json({ status: true, message: "Message updated", data: message });
  } catch (error) {
    console.error("Error updating message:", error);
    res.status(500).json({ status: false, error: "Internal server error" });
  }
};
exports.deleteMessage = async (req, res) => {
  try {
    const { messageId } = req.params;
    const userId = req.user._id; // Use logged-in user

    const message = await Message.findById(messageId);
    if (!message) {
      return res
        .status(404)
        .json({ status: false, error: "Message not found" });
    }

    // Only sender can delete
    if (message.senderId.toString() !== userId.toString()) {
      return res.status(403).json({ status: false, error: "Not authorized" });
    }

    await message.deleteOne();

    // Emit deletion event
    emitToUser(message.receiverId.toString(), "messageDeleted", { messageId });

    res
      .status(200)
      .json({ status: true, message: "Message deleted successfully" });
  } catch (error) {
    console.error("Error deleting message:", error);
    res.status(500).json({ status: false, error: "Internal server error" });
  }
};
