// controllers/lockController.js
const User = require("../models/userModel"); // Assuming you have the User model
const cron = require("node-cron");

// 1. Set Lock Time for User
exports.setLockTime = async (req, res) => {
  const { lockTimeInMinutes } = req.body; // lockTimeInMinutes can be any value from the client

  if (isNaN(lockTimeInMinutes) || lockTimeInMinutes <= 0) {
    return res.status(400).json({
      status: false,
      error: "Invalid lock time. It must be a positive number.",
    });
  }

  const lockDuration = lockTimeInMinutes * 60 * 1000; // Convert minutes to milliseconds

  try {
    const user = req.user; // Access the current logged-in user
    user.lockTime = lockDuration;
    user.lastAccessTime = Date.now(); // Reset last access time
    user.isLocked = false; // Reset locked status
    await user.save();

    res.status(200).json({
      status: true,
      message: `Lock duration set to ${lockTimeInMinutes} minutes.`,
    });
  } catch (err) {
    console.error(err);
    res
      .status(500)
      .json({ status: false, error: "Failed to update lock time" });
  }
};

// 2. Define Cron Job that Runs Every Minute to Check Lock Status
cron.schedule("* * * * *", async () => {
  try {
    const users = await User.find(); // Get all users (or use a specific filter if needed)

    users.forEach(async (user) => {
      if (user.lockTime > 0) {
        const currentTime = Date.now();
        const elapsedTime = currentTime - user.lastAccessTime;

        if (elapsedTime >= user.lockTime && !user.isLocked) {
          user.isLocked = true;
          await user.save();
          console.log(`User ${user._id} is locked due to inactivity.`);
        }
      }
    });
  } catch (err) {
    console.error("Error checking lock status:", err);
  }
});

// 3. Sample Route to Simulate User Activity and Update Last Access Time
exports.updateLastAccessTime = async (req, res) => {
  try {
    const user = req.user; // Access the current logged-in user

    // Update the last access time for the user
    user.lastAccessTime = Date.now();
    await user.save();

    res.status(200).json({
      status: true,
      message: "User action successful, last access time updated.",
    });
  } catch (err) {
    console.error(err);
    res
      .status(500)
      .json({ status: false, error: "Failed to process user action" });
  }
};

// 4. Check Lock Status of the User
exports.checkLockStatus = async (req, res) => {
  try {
    const user = req.user; // Access the current logged-in user

    if (user.isLocked) {
      return res.status(200).json({ status: true, message: "App is locked" });
    } else {
      return res
        .status(200)
        .json({ status: true, message: "App is not locked" });
    }
  } catch (err) {
    console.error(err);
    res
      .status(500)
      .json({ status: false, error: "Failed to check lock status" });
  }
};

// 5. Unlock the App Manually
exports.unlockApp = async (req, res) => {
  try {
    const user = req.user; // Access the current logged-in user

    // Manually unlock the app
    user.isLocked = false;
    await user.save();

    res
      .status(200)
      .json({ status: true, message: "App unlocked successfully." });
  } catch (err) {
    console.error(err);
    res.status(500).json({ status: false, error: "Failed to unlock app" });
  }
};
