const axios = require("axios");
const catchAsync = require("../utils/catchAsync");
const getPlaceDetails = require("../utils/placeHelper");
const User = require("../models/userModel");

// Helper function to calculate distance using the Haversine formula
const calculateDistance = (lat1, lon1, lat2, lon2) => {
  const toRadians = (degrees) => degrees * (Math.PI / 180);
  const R = 6371; // Radius of the Earth in kilometers
  const dLat = toRadians(lat2 - lat1);
  const dLon = toRadians(lon2 - lon1);

  const a =
    Math.sin(dLat / 2) * Math.sin(dLat / 2) +
    Math.cos(toRadians(lat1)) *
      Math.cos(toRadians(lat2)) *
      Math.sin(dLon / 2) *
      Math.sin(dLon / 2);

  const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
  return R * c; // Distance in kilometers
};



exports.updateCurrentLocation = catchAsync(async (req, res, next) => {
  const { latitude, longitude } = req.body;
  const userId = req.user._id; // Authenticated user ID

  try {
    // ✅ Validate latitude and longitude
    if (!latitude || !longitude || isNaN(latitude) || isNaN(longitude)) {
      return res.status(400).json({
        status: false,
        error: "Valid latitude and longitude are required.",
      });
    }

    // ✅ Call helper to get place details
    const { placeId, locationName, isValidPlace, nearestPlace, minDistance } =
      await getPlaceDetails(latitude, longitude);

    // ✅ Update user location in the database
    await User.findByIdAndUpdate(userId, {
      latitude,
      longitude,
      placeId,
      location: locationName,
      isValidPlace,
      distance: minDistance !== Infinity ? minDistance : null, // Save distance if a valid place is found
    });

    // ✅ Send response
    res.status(200).json({
      status: true,
      isValidPlace,
      message: "Location updated successfully.",
      location: {
        latitude,
        longitude,
        placeId,
        locationName,
        distance: minDistance !== Infinity ? minDistance : null, // Include distance in the response
      },
    });
  } catch (error) {
    console.error("❌ Error updating location:", error.message);
    res.status(500).json({
      status: false,
      error: "Something went wrong while updating location.",
    });
  }
});


exports.clearUserLocation = async (req, res) => {
  try {
    const userId = req.user.id; // set by authMiddleware

    const updated = await User.findByIdAndUpdate(
      userId,
      {
        $set: {
          placeId: null,
          longitude: null,
          latitude: null,
          isValidPlace: false, // optional, if you want to reset this too
        },
      },
      { new: true }
    );

    if (!updated) {
      return res.status(404).json({ status: false, message: 'User not found' });
    }

    console.log(`✅ Cleared location for user ${userId}`);

    res.json({
      status: true,
      message: 'Location cleared successfully',
    });
  } catch (err) {
    console.error('❌ Error clearing user location:', err);
    res.status(500).json({
      status: false,
      message: 'Failed to clear user location',
    });
  }
};