const axios = require("axios");
const jwt = require("jsonwebtoken");
const User = require("../models/userModel");
const path = require("path");
const multer = require("multer");
const fs = require("fs");
const sharp = require("sharp");

const LINKEDIN_USER_INFO_URL = "https://api.linkedin.com/v2/me";
const LINKEDIN_EMAIL_URL =
  "https://api.linkedin.com/v2/emailAddress?q=members&projection=(elements*(handle~))";
const USERINFO_URL = "https://api.linkedin.com/v2/userinfo";
const AUTHORIZATION_URL = "https://www.linkedin.com/oauth/v2/authorization";
const TOKEN_URL = "https://www.linkedin.com/oauth/v2/accessToken";

const signToken = (id) => {
  return jwt.sign({ id }, process.env.JWT_SECRET, {
    expiresIn: process.env.JWT_EXPIRES_IN,
  });
};

const createSendToken = async (user, statusCode, req, res) => {
  const token = signToken(user._id);

  // Save the active token to enforce single-device login
  user.activeToken = token;
  await user.save({ validateBeforeSave: false });

  // 1 year in days
  const cookieExpireInDays = 365;

  const cookieOptions = {
    expires: new Date(
      Date.now() + cookieExpireInDays * 24 * 60 * 60 * 1000
    ),
    httpOnly: true,
    secure: req.secure || req.headers["x-forwarded-proto"] === "https",
    sameSite: "strict",
  };

  // Optional: Send cookie if needed
  // res.cookie("jwt", token, cookieOptions);

  // Hide password before sending response
  user.password = undefined;

  res.status(statusCode).json({
    status: true,
    token,
    data: {
      user,
    },
  });
};

// Configure multer for file uploads (same as your original code)
const storage = multer.diskStorage({
  destination: (req, file, cb) => {
    const uploadPath = path.join(__dirname, "uploads", "photos"); // Absolute path
    cb(null, uploadPath);
  },
  filename: (req, file, cb) => {
    const uniqueSuffix = Date.now() + "-" + Math.round(Math.random() * 1e9);
    cb(null, uniqueSuffix + "-" + file.originalname); // Add unique suffix to filename
  },
});

// File filter to only accept image files
const fileFilter = (req, file, cb) => {
  if (file.mimetype.startsWith("image/")) {
    cb(null, true);
  } else {
    cb(new Error("Please upload only images."), false);
  }
};

const upload = multer({
  storage,
  fileFilter,
  limits: {
    fileSize: 5 * 1024 * 1024, // Limit file size to 5MB
  },
});
const sanitizeFilename = (filename) => {
  // Remove query parameters and special characters that are not allowed in filenames
  return filename.replace(/[^a-zA-Z0-9\.\-\_]/g, "_").split("?")[0];
};
// Helper function to download the image from the URL and save it locally
const saveProfilePic = async (imageUrl) => {
  try {
    // Validate the image URL format
    if (!imageUrl || !/^https?:\/\/.*/.test(imageUrl)) {
      console.error("Invalid URL provided:", imageUrl);
      return null; // Return the original URL if invalid
    }

    // Sanitize filename to prevent any issues with invalid characters
    const fileName = sanitizeFilename(
      `${Date.now()}-${Math.floor(Math.random() * 1e8)}-${path.basename(
        imageUrl,
        path.extname(imageUrl)
      )}.png`
    );

    const photoDir = path.join(__dirname, "..", "uploads", "photos");

    // Ensure the directory exists, if not, create it
    if (!fs.existsSync(photoDir)) {
      fs.mkdirSync(photoDir, { recursive: true });
    }

    const filePath = path.join(photoDir, fileName);

    // Download the image and save it locally
    const writer = fs.createWriteStream(filePath);
    const response = await axios({
      url: imageUrl,
      method: "GET",
      responseType: "stream",
    });

    response.data.pipe(writer);

    return new Promise((resolve, reject) => {
      writer.on("finish", () => resolve(`uploads/photos/${fileName}`)); // Resolve with the saved image path
      writer.on("error", (err) => {
        console.error("Error downloading profile picture:", err.message);
        reject(new Error("Error downloading profile picture"));
      });
    });
  } catch (err) {
    console.error("Error downloading profile picture:", err.message);
    // If download fails, return the original URL as a fallback
    return null;
  }
};


const authenticateLinkedIn = async (req, res) => {
  const { access_token, fcmToken } = req.body;
  
  
  if (!access_token) {
    return res
      .status(400)
      .json({ status: false, message: "Access token is missing" });
  }

  try {
    // Fetch user profile data using access_token
    const profileResponse = await axios.get(USERINFO_URL, {
      headers: { Authorization: `Bearer ${access_token}` },
    });

    const profile = profileResponse.data;


    // Create a user object
    const user = {
      sub: profile.sub,
      email_verified: profile.email_verified,
      name: `${profile.given_name} ${profile.family_name}`,
      locale: profile.locale,
      given_name: profile.given_name,
      family_name: profile.family_name,
      email: profile.email,
      loginWith: "linkedin",
      picture: profile.picture, // Profile picture URL
    };

    let existingUser = await User.findOne({ email: profile.email });

    if (existingUser) {
      // Update user if they already exist
      existingUser.name = user.name;
      existingUser.picture = null; // Ensure profile picture is updated

      // Update fcmToken if provided
      if (fcmToken) {
       
        existingUser.fcmToken = fcmToken;
      }

      await existingUser.save();
      return createSendToken(existingUser, 200, req, res);
    } else {
      // Create a new user if they don't exist
      const newUser = new User({
        firstName: user.given_name,
        lastName: user.family_name,
        email: user.email,
        emailVerified: user.email_verified,
        role: "user", // Default to "user"
        loginWith: "linkedin",
        active: true, // Mark the user as active
      });

      // Fetch the profile picture URL from LinkedIn
      const profilePicUrl = user.picture;

      if (profilePicUrl) {
        try {
          // Save the profile picture locally
          const savedPicPath = await saveProfilePic(profilePicUrl);
          newUser.photo = savedPicPath; // Save the path of the saved photo
        } catch (err) {
          console.error("Error saving profile picture:", err.message);
          return res.status(500).json({
            status: false,
            error: "Error saving profile picture",
          });
        }
      }

      // Save fcmToken if provided
      if (fcmToken) {
          
        newUser.fcmToken = fcmToken;
      }

      await newUser.save(); // Save the new user to the database
      return createSendToken(newUser, 201, req, res); // Generate and send JWT token
    }
  } catch (error) {
    console.error("Error in LinkedIn callback:", error.message);
    return res.status(500).json({
      status: false,
      error: "Failed to complete LinkedIn authentication.",
      details: error.message,
    });
  }
};



module.exports = {
  authenticateLinkedIn,
};
