const InterestTab = require("../models/interestsModel");
const User = require("../models/userModel");

// Create a new Interest Tab
exports.createInterestTab = async (req, res) => {
  try {
    const { topCategory, interests } = req.body;

    // Check if category and interests are provided
    if (!topCategory || !interests || !Array.isArray(interests)) {
      return res.status(400).json({
        status: false,
        error: "Category and interests must be provided.",
      });
    }

    // Check if an InterestTab with the given topCategory already exists
    const existingInterestTab = await InterestTab.findOne({ topCategory });

    // Filter out duplicates in the interests array by title
    const uniqueInterests = [
      ...new Map(
        interests.map((interest) => [interest.title, interest])
      ).values(),
    ];

    if (existingInterestTab) {
      // Filter out interests that already exist in the InterestTab
      const newInterests = uniqueInterests.filter(
        (interest) =>
          !existingInterestTab.interests.some(
            (existingInterest) => existingInterest.title === interest.title
          )
      );

      // If there are no new interests to add, return a message
      if (newInterests.length === 0) {
        return res.status(400).json({
          status: false,
          error: "All provided interests already exist in this category.",
        });
      }

      // Add the new interests to the existing InterestTab
      existingInterestTab.interests.push(...newInterests);
      await existingInterestTab.save();

      return res.status(200).json({
        status: true,
        message: "Interest Tab updated successfully with new interests.",
        data: existingInterestTab,
      });
    } else {
      // If no InterestTab exists, create a new one
      const newInterestTab = new InterestTab({
        topCategory,
        interests: uniqueInterests,
      });
      await newInterestTab.save();

      return res.status(201).json({
        status: true,
        message: "Interest Tab created successfully.",
        data: newInterestTab,
      });
    }
  } catch (error) {
    console.error(error);
    return res.status(500).json({
      status: false,
      error: "An error occurred while creating the Interest Tab.",
    });
  }
};

// Get all Interest Tabs
exports.getAllInterestTabs = async (req, res) => {
  try {
    const interestTabs = await InterestTab.find();

    if (!interestTabs.length) {
      return res.status(404).json({
        status: false,
        message: "No Interest Tabs found.",
      });
    }

    return res.status(200).json({
      status: true,
      data: interestTabs,
    });
  } catch (error) {
    console.error(error);
    return res.status(500).json({
      status: false,
      error: "An error occurred while retrieving the Interest Tabs.",
    });
  }
};

// Get a specific Interest Tab by category
exports.getInterestTabByCategory = async (req, res) => {
  try {
    const { topCategory } = req.params;

    const interestTab = await InterestTab.findOne({ topCategory });

    if (!interestTab) {
      return res.status(404).json({
        status: false,
        message: `Interest Tab with category '${topCategory}' not found.`,
      });
    }

    return res.status(200).json({
      status: true,
      data: interestTab,
    });
  } catch (error) {
    console.error(error);
    return res.status(500).json({
      status: false,
      error: "An error occurred while retrieving the Interest Tab.",
    });
  }
};

// Update the interests in an existing Interest Tab by category
exports.updateInterestTab = async (req, res) => {
  try {
    const { topCategory } = req.params;
    const { interests } = req.body;

    // Check if interests are provided as an array
    if (!interests || !Array.isArray(interests)) {
      return res.status(400).json({
        status: false,
        error: "Interests must be provided as an array.",
      });
    }

    // Find the InterestTab by topCategory
    const interestTab = await InterestTab.findOne({ topCategory });

    if (!interestTab) {
      return res.status(404).json({
        status: false,
        message: `Interest Tab with category '${topCategory}' not found.`,
      });
    }

    // Get the titles of the current interests in the interestTab
    const currentInterestsTitles = interestTab.interests.map(
      (interest) => interest.title
    );

    // Filter out new interests that already exist in the current interests based on the 'title'
    const newInterests = interests.filter((newInterest) => {
      return (
        newInterest.title && !currentInterestsTitles.includes(newInterest.title)
      );
    });

    // Add the new interests to the existing ones
    interestTab.interests = [...interestTab.interests, ...newInterests];

    // Save the updated InterestTab
    await interestTab.save();

    return res.status(200).json({
      status: true,
      message: "Interest Tab updated successfully.",
      data: interestTab,
    });
  } catch (error) {
    console.error(error);
    return res.status(500).json({
      status: false,
      error: "An error occurred while updating the Interest Tab.",
    });
  }
};

// Delete an Interest Tab by category
exports.deleteInterestTab = async (req, res) => {
  try {
    const { topCategory } = req.params;

    const interestTab = await InterestTab.findOneAndDelete({ topCategory });

    if (!interestTab) {
      return res.status(404).json({
        status: false,
        message: `Interest Tab with category '${topCategory}' not found.`,
      });
    }

    return res.status(200).json({
      status: true,
      message: "Interest Tab deleted successfully.",
    });
  } catch (error) {
    console.error(error);
    return res.status(500).json({
      status: false,
      error: "An error occurred while deleting the Interest Tab.",
    });
  }
};
// Function to search interests with advanced features
exports.SearchInterest = async (req, res) => {
  try {
    const {
      query, // Search query for interest titles (e.g., "Technology")
      topCategory, // Optional filter for topCategory
      page = 1, // Default to page 1 for pagination
      limit = 10, // Default to 10 results per page
      sortBy = "createdAt", // Default to sorting by creation date
      sortOrder = "desc", // Default to descending order
    } = req.query;

    // Check if the query parameter is provided
    if (!query) {
      return res.status(400).json({
        status: false,
        error: "Search query is required.",
      });
    }

    // Build the search filter
    const searchFilter = {
      "interests.title": { $regex: query, $options: "i" }, // Case-insensitive search
    };

    // Optional: Filter by topCategory if provided
    if (topCategory) {
      searchFilter.topCategory = topCategory;
    }

    // Pagination calculations
    const skip = (page - 1) * limit;

    // Sort by the specified field and order
    const sort = {};
    sort[sortBy] = sortOrder === "asc" ? 1 : -1; // 1 for ascending, -1 for descending

    // Perform the search query with pagination, sorting, and filters
    const interestTabs = await InterestTab.find(searchFilter)
      .skip(skip)
      .limit(Number(limit))
      .sort(sort)
      .select("topCategory interests createdAt");

    // Check if any results are found
    if (interestTabs.length === 0) {
      return res.status(404).json({
        status: false,
        message: "No interest tabs found matching the query.",
      });
    }

    // Count the total number of matching interest tabs for pagination info
    const totalCount = await InterestTab.countDocuments(searchFilter);

    // Return the paginated, sorted results
    res.status(200).json({
      status: true,
      message: "Interest tabs found.",
      data: interestTabs,
      pagination: {
        totalCount,
        totalPages: Math.ceil(totalCount / limit),
        currentPage: page,
        perPage: limit,
      },
    });
  } catch (error) {
    console.error("Error in advancedSearchInterest:", error.message);
    res.status(500).json({
      status: false,
      error: "An error occurred while searching for interests.",
    });
  }
};
// Function to get the most popular interests and their user count, with top 3 interests
exports.getTop3PopularInterests = async (req, res) => {
  try {
    // Step 1: Aggregate all user interests
    const users = await User.find({}).select("interests"); // Fetch all users and only their interests

    // Step 2: Count the occurrences of each interest
    const interestCount = {};

    users.forEach((user) => {
      user.interests.forEach((interest) => {
        // Increase the count for the interest if it already exists
        if (interestCount[interest]) {
          interestCount[interest]++;
        } else {
          // Initialize the count for the interest if it's the first time
          interestCount[interest] = 1;
        }
      });
    });

    // Step 3: Convert interest count object into an array of interests and their counts
    const popularInterests = Object.keys(interestCount).map((interest) => ({
      title: interest,
      userCount: interestCount[interest],
    }));

    // Step 4: Sort interests based on userCount in descending order
    popularInterests.sort((a, b) => b.userCount - a.userCount);

    // Step 5: Extract top 3 interests
    const top3Interests = popularInterests.slice(0, 3);

    // Step 6: Return the top 3 most popular interests
    res.status(200).json({
      status: true,
      message: "Top 3 most popular interests fetched successfully.",
      data: top3Interests,
    });
  } catch (error) {
    console.error("Error in getTop3PopularInterests:", error.message);
    res.status(500).json({
      status: false,
      error: "An error occurred while fetching top 3 popular interests.",
    });
  }
};
