const Message = require("../models/messageModel");
const {
  io,
  getReceiverSocketId,
  getRoomName,
  activeRooms,
  userSocketMap,
  emitToUser,
} = require("../utils/socket.js");
const mongoose = require("mongoose");
const User = require("../models/userModel"); // User model
const admin = require("../utils/firebaseServices");
const { createAndEmitNotification } = require("../utils/notificationUtils"); // Import the notification handler

exports.sendImages = async (req, res) => {
  try {
    const senderId = req.user._id;
    const { receiverId } = req.body;

    // Validate input
    if (!req.files || req.files.length === 0) {
      return res.status(400).json({
        status: false,
        message: "No images uploaded",
      });
    }

    if (req.files.length > 5) {
      return res.status(400).json({
        status: false,
        message: "Maximum 5 images allowed",
      });
    }

    if (
      !mongoose.Types.ObjectId.isValid(senderId) ||
      !mongoose.Types.ObjectId.isValid(receiverId)
    ) {
      return res.status(400).json({
        status: false,
        message: "Invalid user IDs",
      });
    }

    // Fetch users in parallel
    const [receiver, sender] = await Promise.all([
      User.findById(receiverId)
        .select("fcmToken privacyPreferences notifications blockedUsers")
        .lean(),
      User.findById(senderId).select("firstName lastName photo").lean(),
    ]);

    // Validate users
    if (!receiver || !sender) {
      return res.status(404).json({
        status: false,
        message: "User not found",
      });
    }

    if (receiver.blockedUsers?.includes(senderId)) {
      return res.status(403).json({
        status: false,
        message: "You have been blocked by this user",
      });
    }

    if (!receiver.privacyPreferences?.allowMessages) {
      return res.status(403).json({
        status: false,
        message: "This user does not allow messages",
      });
    }

    // Check if receiver is active in the room
    const roomName = getRoomName(senderId, receiverId);
    const isReceiverActive = activeRooms[roomName]?.users.has(receiverId);

    // Create message with proper initial status
    const newMessage = new Message({
      senderId,
      receiverId,
      images: req.files.map((file) => `/uploads/photos/${file.filename}`),
      content: "Image message",
      seen: isReceiverActive,
    });

    await newMessage.save();

    // Prepare response
    const responseData = {
      status: true,
      message: "Images sent successfully",
      data: newMessage.toObject(),
    };

    // Defer real-time updates
    process.nextTick(async () => {
      try {
        const receiverSocketId = getReceiverSocketId(receiverId);

        // Real-time updates if receiver is online
        if (receiverSocketId) {
          // Emit new message
          // Send new message event
          emitToUser(receiverId, "newMessage", newMessage);

          // Update conversation list
          const unreadCount = await Message.countDocuments({
            senderId,
            receiverId,
            seen: false,
          });

          emitToUser(receiverId, "getUnreadMessages", {
            _id: senderId,
            firstName: sender.firstName,
            lastName: sender.lastName,
            photo: sender.photo,
            lastMessage: "Image message",
            lastMessageDateTime: newMessage.createdAt.toISOString(),
            unseenCount: isReceiverActive ? unreadCount : unreadCount + 1,
            online_status: isUserOnline(senderId),
          });
        }

        // Send push notification if receiver is offline
        if (!isReceiverActive && receiver.fcmToken) {
          const notification = {
            notification: {
              title: `${sender.firstName} ${sender.lastName}`,
              body: "📷 Sent you images",
              image: sender.photo,
            },
            data: {
              type: "new_message",
              senderId: senderId.toString(),
              receiverId: receiverId.toString(),
              messageId: newMessage._id.toString(),
              click_action: "FLUTTER_NOTIFICATION_CLICK",
              url: `/messages/${senderId}`,
            },
            token: receiver.fcmToken,
          };

          await admin
            .messaging()
            .send(notification)
            .catch((err) => console.error("FCM error:", err));
        }
      } catch (err) {
        console.error("Background processing error:", err);
      }
    });

    res.status(200).json(responseData);
  } catch (err) {
    console.error("Error sending images:", err);

    // Clean up uploaded files if error occurred
    if (req.files) {
      await Promise.all(
        req.files.map((file) =>
          fs
            .unlink(file.path)
            .catch((cleanupErr) =>
              console.error("File cleanup error:", cleanupErr)
            )
        )
      );
    }

    res.status(500).json({
      status: false,
      message: "Failed to send images",
      error: process.env.NODE_ENV === "development" ? err.message : undefined,
    });
  }
};
