const jwt = require("jsonwebtoken");
const axios = require("axios");
const path = require("path");
const fs = require("fs");
const User = require("../models/userModel");
const { OAuth2Client } = require("google-auth-library");

const client = new OAuth2Client();

const IOS_CLIENT_ID = process.env.IOS_CLIENT_ID;
const ANDROID_CLIENT_ID = process.env.ANDROID_CLIENT_ID;

// Sign JWT token
const signToken = (id) => {
  return jwt.sign({ id }, process.env.JWT_SECRET, {
    expiresIn: process.env.JWT_EXPIRES_IN,
  });
};

// Send token in cookie + response
const createSendToken = async (user, statusCode, req, res) => {
  const token = signToken(user._id);

  // Save the active token to enforce single-device login
  user.activeToken = token;
  await user.save({ validateBeforeSave: false });

  // 1 year in days
  const cookieExpireInDays = 365;

  const cookieOptions = {
    expires: new Date(
      Date.now() + cookieExpireInDays * 24 * 60 * 60 * 1000
    ),
    httpOnly: true,
    secure: req.secure || req.headers["x-forwarded-proto"] === "https",
    sameSite: "strict",
  };

  // Optional: Send cookie if needed
  // res.cookie("jwt", token, cookieOptions);

  // Hide password before sending response
  user.password = undefined;

  res.status(statusCode).json({
    status: true,
    token,
    data: {
      user,
    },
  });
};


// Download and save profile picture
const saveProfilePic = async (imageUrl) => {
  const fileName = `${Date.now()}-${Math.floor(Math.random() * 1e8)}.png`;
  const filePath = path.join(__dirname, "..", "uploads", "photos", fileName);

  const writer = fs.createWriteStream(filePath);
  const response = await axios({
    url: imageUrl,
    method: "GET",
    responseType: "stream",
  });

  response.data.pipe(writer);

  return new Promise((resolve, reject) => {
    writer.on("finish", () => resolve(`uploads/photos/${fileName}`));
    writer.on("error", (err) => reject(new Error("Image download error: " + err.message)));
  });
};

// Fetch user info using access token
const getUserInfoWithAccessToken = async (accessToken) => {
  const response = await axios.get("https://www.googleapis.com/oauth2/v3/userinfo", {
    headers: { Authorization: `Bearer ${accessToken}` },
  });
  return response.data;
};

// Authenticate via Google (accepts either id_token or access_token)
const authenticateGoogle = async (req, res) => {
  const { access_token, id_token, fcmToken } = req.body; // <-- accept fcmToken

  if (!access_token && !id_token) {
    return res.status(400).send("Access token or ID token is required");
  }

  try {
    let userInfo;

    if (id_token) {
      // Verify ID token (stricter)
      const ticket = await client.verifyIdToken({
        idToken: id_token,
        audience: [ANDROID_CLIENT_ID, IOS_CLIENT_ID],
      });

      const payload = ticket.getPayload();
      userInfo = {
        email: payload.email,
        email_verified: payload.email_verified,
        given_name: payload.given_name,
        family_name: payload.family_name,
        picture: payload.picture,
      };
    } else {
      // Fallback to access_token method
      userInfo = await getUserInfoWithAccessToken(access_token);
    }

    const profilePicUrl = userInfo.picture;
    let user = await User.findOne({ email: userInfo.email });

    if (user) {
      // Update existing user
      user.firstName = userInfo.given_name;
      user.lastName = userInfo.family_name;
      user.emailVerified = userInfo.email_verified;
      user.loginWith = "google";
      user.active = true;

      // ✅ update fcmToken if provided
      if (fcmToken) {
        user.fcmToken = fcmToken;
      }

      await user.save();
      return createSendToken(user, 200, req, res);
    } else {
      // Create new user
      const newUser = new User({
        firstName: userInfo.given_name,
        lastName: userInfo.family_name,
        email: userInfo.email,
        emailVerified: userInfo.email_verified,
        role: "user",
        loginWith: "google",
        active: true,
        fcmToken: fcmToken || null,  // ✅ save fcmToken for new users
      });

      if (profilePicUrl) {
        const picPath = await saveProfilePic(profilePicUrl);
        newUser.photo = picPath;
      }

      await newUser.save();
      return createSendToken(newUser, 201, req, res);
    }
  } catch (error) {
    console.error("❌ Google Auth Error:", error.message);
    return res.status(400).send("Authentication failed: " + error.message);
  }
};


module.exports = {
  authenticateGoogle,
};
