const User = require("../models/userModel"); // User model
const Connection = require("../models/connectionModel"); // Connection model
const mongoose = require("mongoose");
const axios = require("axios");
const moment = require("moment");
const { io, getReceiverSocketId ,emitToUser} = require("../utils/socket");
const { createAndEmitNotification } = require("../utils/notificationUtils");
const admin = require("../utils/firebaseServices");

// Controller to send a connection request
exports.sendConnectionRequest = async (req, res) => {
  try {
    const userId = req.user._id; // Logged-in user ID
    const { targetUserId } = req.body; // Target user ID
console.log(userId);
console.log(targetUserId);
    // Fetch the logged-in user with credit balance
    const user = await User.findById(userId).select(
      "firstName lastName credits"
    );
    if (!user) {
      return res.status(404).json({
        status: false,
        error: "User not found.",
      });
    }

    // Check if user has enough credits
    if (user.credits <= 0) {
      return res.status(400).json({
        status: false,
        error: "Insufficient credits to send a connection request.",
      });
    }

    // Prevent self-connection requests
    if (userId.toString() === targetUserId.toString()) {
      return res.status(400).json({
        status: false,
        error: "You cannot send a connection request to yourself.",
      });
    }

    // Fetch target user with privacy preferences and notifications
    const targetUser = await User.findById(targetUserId).select(
      "fcmToken privacyPreferences notifications"
    );
    if (!targetUser) {
      return res.status(404).json({
        status: false,
        error: "Target user not found.",
      });
    }

    // Check if the target user allows meet-up requests
    if (!targetUser.privacyPreferences.allowMeetUpRequests) {
      return res.status(403).json({
        status: false,
        error: "This user does not allow connection requests.",
      });
    }

    // Check if a connection already exists
    const existingConnection = await Connection.findOne({
      $or: [
        { sender: userId, receiver: targetUserId },
        { sender: targetUserId, receiver: userId },
      ],
    });

    if (existingConnection) {
      return res.status(400).json({
        status: false,
        error: "Connection request already sent or already connected.",
      });
    }

    // Save connection request
    const newConnectionRequest = new Connection({
      sender: userId,
      receiver: targetUserId,
      status: "pending",
    });

    // Update the target user's status if needed
    await User.findByIdAndUpdate(targetUserId, { $set: { status: "pending" } });

    // Respond to the client immediately
    res.status(200).json({
      status: true,
      message: "Connection request sent successfully.",
      data: {
        senderId: userId,
        receiverId: targetUserId,
        requestId: newConnectionRequest._id,
        link: `/connections/${newConnectionRequest._id}`,
      },
    });

    // Use process.nextTick to defer notification logic
    process.nextTick(async () => {
      try {
        // Check if the target user has enabled notifications for new meet-ups
        if (targetUser.notifications.newMeetUps) {
          // Create a socket notification
          const notificationMessage = `${user.firstName} ${user.lastName} has sent you a connection request.`;
          const notificationLink = `/connections/${newConnectionRequest._id}`;
          const senderId = userId;
          await newConnectionRequest.save();
          emitToUser(targetUserId, "sendConnectionRequest", {
            title: "New Connection Request",
            userid: targetUserId,
            message: notificationMessage,
            link: notificationLink,
          });

          await createAndEmitNotification(
            targetUserId,
            notificationMessage,
            "Send Connection",
            notificationLink,
            senderId
          );

          // Send Push Notification via Firebase with sender & receiver ID
          if (targetUser.fcmToken) {
            const fcmPayload = {
              notification: {
                title: "New Connection Request",
                body: notificationMessage,
              },
              data: {
                senderId: senderId.toString(), // Include sender ID
                receiverId: targetUserId.toString(), // Include receiver ID
                click_action: "FLUTTER_NOTIFICATION_CLICK",
                url: notificationLink, // Clickable link for redirection
              },
              token: targetUser.fcmToken,
            };

            await admin.messaging().send(fcmPayload);
          }
        }
      } catch (notificationError) {
        console.error(
          "Error during notification processing:",
          notificationError
        );
      }
    });
  } catch (error) {
    console.error("Error sending connection request:", error);
    res.status(500).json({
      status: false,
      error: "An error occurred while sending the connection request.",
    });
  }
};

exports.acceptConnectionRequest = async (req, res) => {
  try {
    const userId = req.user._id; // Logged-in user ID from middleware
    const { connectionId } = req.body; // Connection request ID

    // Validate connection request
    const connection = await Connection.findById(connectionId).populate(
      "sender receiver",
      "firstName lastName email photo fcmToken credits"
    );
    if (!connection) {
      return res.status(404).json({
        status: false,
        message: "Connection request not found.",
      });
    }

    // Ensure only the receiver can accept the request
    if (connection.receiver._id.toString() !== userId.toString()) {
      return res.status(403).json({
        status: false,
        message: "You are not authorized to accept this connection request.",
      });
    }

    // Deduct credit from the sender (not the receiver)
    if (connection.sender.credits < 1) {
      return res.status(400).json({
        status: false,
        message: "Sender does not have enough credits to proceed.",
      });
    }
    connection.sender.credits -= 1; // Deduct 1 credit from the sender
    await connection.sender.save(); // Save the updated sender

    // Update the connection status to 'accepted'
    connection.status = "accepted";
    await connection.save();

    // Update user statuses and connections
    await Promise.all([
      User.findByIdAndUpdate(connection.sender._id, {
        $addToSet: { connections: connection.receiver._id },
        $inc: { totalConnections: 1 }, // Increment sender's total connections
      }),
      User.findByIdAndUpdate(connection.receiver._id, {
        $addToSet: { connections: connection.sender._id },
        $inc: { totalConnections: 1 }, // Increment receiver's total connections
      }),
    ]);

    // Use process.nextTick to defer notification logic
    process.nextTick(async () => {
      // Emit real-time update to the sender
      emitToUser(connection.sender._id.toString(), "connection_accepted", {
        connectionId: connection._id,
        receiver: {
          _id: connection.receiver._id,
          firstName: connection.receiver.firstName,
          lastName: connection.receiver.lastName,
          email: connection.receiver.email,
          photo: connection.receiver.photo,
        },
      });

      // Create and emit notification for the sender
      const notificationMessage = `${connection.receiver.firstName} ${connection.receiver.lastName} has accepted your connection request.`;
      const notificationLink = `/connections/${connection._id}`;

      await createAndEmitNotification(
        connection.sender._id,
        notificationMessage,
        "Accept Connection",
        notificationLink,
        userId
      );

      // ✅ Send Push Notification via Firebase with sender & receiver ID
      if (connection.sender.fcmToken) {
        const fcmPayload = {
          notification: {
            title: "Connection Request Accepted",
            body: notificationMessage,
          },
          data: {
            senderId: connection.receiver._id.toString(), // Include sender ID
            receiverId: connection.sender._id.toString(), // Include receiver ID
            click_action: "FLUTTER_NOTIFICATION_CLICK",
            url: notificationLink, // Clickable link for redirection
          },
          token: connection.sender.fcmToken,
        };

        try {
          await admin.messaging().send(fcmPayload);
        } catch (fcmError) {
          console.error("Error sending FCM notification:", fcmError);
        }
      }
    });

    // Send response to the receiver
    res.status(200).json({
      status: true,
      message: "Connection request accepted successfully.",
      connection: {
        _id: connection._id,
        sender: {
          _id: connection.sender._id,
          firstName: connection.sender.firstName,
          lastName: connection.sender.lastName,
          email: connection.sender.email,
          photo: connection.sender.photo,
        },
        receiver: {
          _id: connection.receiver._id,
          firstName: connection.receiver.firstName,
          lastName: connection.receiver.lastName,
          email: connection.receiver.email,
          photo: connection.receiver.photo,
        },
        status: connection.status,
      },
    });
  } catch (error) {
    console.error("Error while accepting connection request:", error);
    res.status(500).json({
      status: false,
      message: "An error occurred while accepting the connection request.",
    });
  }
};

// Controller to reject a connection request
exports.rejectConnectionRequest = async (req, res) => {
  try {
    const userId = req.user._id; // Logged-in user
    const { connectionId } = req.body; // Connection request ID

    // Find connection
    const connection = await Connection.findById(connectionId).populate(
      "sender receiver",
      "firstName lastName email photo fcmToken"
    );

    if (!connection) {
      return res.status(404).json({
        status: false,
        error: "Connection request not found.",
      });
    }

    // Only the receiver can reject
    if (connection.receiver._id.toString() !== userId.toString()) {
      return res.status(403).json({
        status: false,
        error: "You are not authorized to reject this connection request.",
      });
    }

    // Delete the connection request
    await Connection.findByIdAndDelete(connectionId);

    // ✅ Emit real-time event to the sender
    emitToUser(connection.sender._id.toString(), "Connection Rejected", {
      receiver: userId,
      connectionId,
    });

    // ✅ Create + emit notification for the sender
    const notificationMessage = `${connection.receiver.firstName} ${connection.receiver.lastName} has rejected your connection request.`;
    const notificationLink = `/connections/${connectionId}`;

    // await createAndEmitNotification(
    //   connection.sender._id,
    //   notificationMessage,
    //   "reject_connection",   // use valid type from schema
    //   notificationLink,
    //   connection.receiver._id
    // );

    res.status(200).json({
      status: true,
      message: "Connection request rejected and removed successfully.",
    });
  } catch (error) {
    console.error("❌ Error while rejecting connection request:", error);
    res.status(500).json({
      status: false,
      error: "An error occurred while rejecting the connection request.",
    });
  }
};

exports.getUserConnections = async (req, res) => {
  try {
    const userId = req.user._id;

    // Find all accepted connections
    const connections = await Connection.find({
      $or: [{ sender: userId }, { receiver: userId }],
      status: "accepted",
    }).select("sender receiver -_id");

    // Extract connection user IDs
    const connectionUserIds = connections.map((conn) =>
      conn.sender.toString() === userId.toString() ? conn.receiver : conn.sender
    );

    // Fetch all details of connected users (excluding sensitive fields)
    const connectedUsers = await User.find({
      _id: { $in: connectionUserIds },
    }).select("-password -otps -resetToken -resetTokenExpire -__v");

    // Add online status
    const connectedUsersWithStatus = connectedUsers.map((user) => ({
      ...user._doc,
      online_status: !!getReceiverSocketId(user._id),
    }));

    res.status(200).json({
      status: true,
      connections: connectedUsersWithStatus,
    });
  } catch (err) {
    console.error("Error fetching user connections:", err);
    res.status(500).json({
      status: false,
      error: "Something went wrong while fetching connections.",
    });
  }
};


exports.getConnectionRequests = async (req, res) => {
  try {
    const currentUserId = req.user._id; // Logged-in user
    console.log("Current User ID:", currentUserId);

    // Validate ObjectId
    if (!mongoose.Types.ObjectId.isValid(currentUserId)) {
      return res.status(400).json({
        status: false,
        error: "Invalid User ID format.",
      });
    }

    // Clean query parameters
    const queryObj = { ...req.query };
    ["page", "sort", "limit", "fields"].forEach((el) => delete queryObj[el]);

    // Advanced filtering (gte, lte, etc.)
    let queryStr = JSON.stringify(queryObj);
    queryStr = queryStr.replace(
      /\b(gte|gt|lte|lt)\b/g,
      (match) => `$${match}`
    );
    const filters = JSON.parse(queryStr);

    // Build main query
    let query = Connection.find({
       receiver: currentUserId,
      status: "pending",
      ...filters,
    })
      .populate("sender", "-password -otps -resetToken -resetTokenExpire -__v")
      .populate("receiver", "-password -otps -resetToken -resetTokenExpire -__v");

    // Sorting
    if (req.query.sort) {
      const sortBy = req.query.sort.split(",").join(" ");
      query = query.sort(sortBy);
    } else {
      query = query.sort({ createdAt: -1 }); // Default: newest first
    }

    // Field limiting
    if (req.query.fields) {
      const fields = req.query.fields.split(",").join(" ");
      query = query.select(fields);
    } else {
      query = query.select("-__v");
    }

    // Pagination
    const page = parseInt(req.query.page, 10) || 1;
    const limit = parseInt(req.query.limit, 10) || 10;
    const skip = (page - 1) * limit;
    query = query.skip(skip).limit(limit);

    // Execute query
    const connectionRequests = await query;

    // Format requests
    const formattedRequests = connectionRequests.map((request) => {
      const calculateDuration = (timestamp) => {
        const now = moment();
        const time = moment(timestamp);
        const duration = moment.duration(now.diff(time));

        if (duration.asMinutes() < 60) {
          return `${Math.floor(duration.asMinutes())} mins ago`;
        } else if (duration.asHours() < 24) {
          return `${Math.floor(duration.asHours())} hours ago`;
        } else if (duration.asDays() < 30) {
          return `${Math.floor(duration.asDays())} days ago`;
        } else if (duration.asMonths() < 12) {
          return `${Math.floor(duration.asMonths())} months ago`;
        } else {
          return `${Math.floor(duration.asYears())} years ago`;
        }
      };

      // Safe status check (handle populated sender/receiver)
      const senderId =
        request.sender?._id?.toString() || request.sender?.toString();
      const receiverId =
        request.receiver?._id?.toString() || request.receiver?.toString();

      let status = "";
      if (senderId === currentUserId.toString()) {
        status = "pending"; // you sent request
      } else if (receiverId === currentUserId.toString()) {
        status = "requested"; // you received request
      }

      return {
        ...request.toObject(),
        createdAt: calculateDuration(request.createdAt),
        updatedAt: calculateDuration(request.updatedAt),
        status,
      };
    });

    // Send response
    res.status(200).json({
      status: true,
      results: formattedRequests.length,
      data: {
        connectionRequests: formattedRequests,
      },
    });
  } catch (err) {
    console.error("Error fetching connection requests:", err);
    res.status(500).json({
      status: false,
      error: "An error occurred while fetching connection requests.",
      details: err.message,
    });
  }
};

exports.cancelConnectionRequest = async (req, res) => {
  try {
    // Check if req.user exists and extract userId
    if (!req.user || !req.user._id) {
      return res.status(401).json({
        status: false,
        error: "Authentication failed. Please log in.",
      });
    }

    const senderId = req.user._id; // Get the logged-in user's ID from the request
    const { receiverId } = req.params; // Receiver ID from the request parameters

    // Validate the receiverId to ensure it's a valid MongoDB ObjectId
    if (!mongoose.Types.ObjectId.isValid(receiverId)) {
      return res.status(400).json({
        status: false,
        error: "Invalid Receiver ID format.",
      });
    }

    // Find and delete the connection request
    const connection = await Connection.findOneAndDelete({
      sender: senderId,
      receiver: receiverId,
      status: "pending", // Only allow canceling pending requests
    });

    // if (!connection) {
    //   return res.status(404).json({
    //     status: false,
    //     error: "No pending connection request found to cancel.",
    //   });
    // }

    // Respond with a success message
    return res.status(200).json({
      status: true,
      message: "Connection request canceled successfully.",
    });
  } catch (err) {
    console.error("Error canceling connection request:", err);
    return res.status(500).json({
      status: false,
      error:
        "An internal server error occurred while canceling the connection request.",
    });
  }
};
exports.getMutualConnections = async (req, res) => {
  try {
    const currentUserId = req.user._id;

    // Fetch targeted user ID from get-suggested-users API
    const suggestedUsersResponse = await axios.get(
      `${process.env.BASE_URL}/api/v1/users/get-suggested-users`
    );

    if (!suggestedUsersResponse.data || !suggestedUsersResponse.data.status) {
      return res.status(500).json({
        status: false,
        message: "Failed to fetch suggested users",
      });
    }

    const suggestedUsers = suggestedUsersResponse.data.data.users || [];

    if (suggestedUsers.length === 0) {
      return res.status(404).json({
        status: false,
        message: "No suggested users found",
      });
    }

    const targetUserId = suggestedUsers[0]._id; // Selecting the first suggested user as target

    if (!mongoose.Types.ObjectId.isValid(targetUserId)) {
      return res.status(400).json({
        status: false,
        message: "Invalid user ID format",
      });
    }

    // Check if target user exists
    const targetUser = await User.findById(targetUserId);
    if (!targetUser) {
      return res.status(404).json({
        status: false,
        message: "Target user not found",
      });
    }

    // Get current user's connections
    const currentUserConnections = await Connection.aggregate([
      {
        $match: {
          $or: [{ user1: currentUserId }, { user2: currentUserId }],
          status: "accepted",
        },
      },
      {
        $project: {
          connectedUserId: {
            $cond: [{ $eq: ["$user1", currentUserId] }, "$user2", "$user1"],
          },
        },
      },
      {
        $group: {
          _id: null,
          userIds: { $addToSet: "$connectedUserId" },
        },
      },
    ]);

    // Get target user's connections
    const targetUserConnections = await Connection.aggregate([
      {
        $match: {
          $or: [{ user1: targetUserId }, { user2: targetUserId }],
          status: "accepted",
        },
      },
      {
        $project: {
          connectedUserId: {
            $cond: [{ $eq: ["$user1", targetUserId] }, "$user2", "$user1"],
          },
        },
      },
      {
        $group: {
          _id: null,
          userIds: { $addToSet: "$connectedUserId" },
        },
      },
    ]);

    // Extract user IDs
    const currentUserIds = currentUserConnections[0]?.userIds || [];
    const targetUserIds = targetUserConnections[0]?.userIds || [];

    // Find mutual connections
    const mutualConnectionIds = currentUserIds.filter((id) =>
      targetUserIds.some((targetId) => targetId.equals(id))
    );

    // Get mutual users details
    const mutualConnections = await User.find(
      { _id: { $in: mutualConnectionIds } },
      "firstName lastName email profilePhoto currentPosition"
    ).lean();

    res.status(200).json({
      status: true,
      count: mutualConnections.length,
      mutualConnections,
    });
  } catch (error) {
    console.error("Error getting mutual connections:", error);
    res.status(500).json({
      status: false,
      message: "Server error",
      error: error.message,
    });
  }
};
