const axios = require("axios");
const mongoose = require("mongoose"); // Import mongoose
const upload = require("../utils/multerConfig");
const crypto = require("crypto");
const { promisify } = require("util");
const jwt = require("jsonwebtoken");
const User = require("./../models/userModel");
const catchAsync = require("./../utils/catchAsync");
const AppError = require("./../utils/appError");
const Email = require("../utils/email");
const Message = require("../models/messageModel");
const Connection = require("../models/connectionModel");
const bcrypt = require("bcryptjs");
const TempUser = require("../models/tempUserModel"); // Temporary user model
const { body, validationResult } = require("express-validator");
const { generateToken } = require("../utils/utils");
const getPlaceDetails = require("../utils/placeHelper");
const {
  generateOTP,
  sendOTPEmailForVerification,
} = require("../utils/otpService");
const { isFloatLocales } = require("validator");

const signToken = (id) => {
  return jwt.sign({ id }, process.env.JWT_SECRET, {
    expiresIn: process.env.JWT_EXPIRES_IN,
  });
};

const createSendToken = async (user, statusCode, req, res) => {
  const token = signToken(user._id);

  // Save the active token to enforce single-device login
  user.activeToken = token;
  await user.save({ validateBeforeSave: false });

  // 1 year in days
  const cookieExpireInDays = 365;

  const cookieOptions = {
    expires: new Date(
      Date.now() + cookieExpireInDays * 24 * 60 * 60 * 1000
    ),
    httpOnly: true,
    secure: req.secure || req.headers["x-forwarded-proto"] === "https",
    sameSite: "strict",
  };

  // Optional: Send cookie if needed
  // res.cookie("jwt", token, cookieOptions);

  // Hide password before sending response
  user.password = undefined;

  res.status(statusCode).json({
    status: true,
    token,
    data: {
      user,
    },
  });
};


exports.signupName = [
  // Validate firstName and lastName
  body("firstName")
    .trim()
    .notEmpty()
    .withMessage("Please enter your last name.")
    .isAlpha("en-US", { ignore: " " }) // Allow alphabets and spaces
    .withMessage("Last name can only contain alphabets and spaces."),

  body("lastName")
    .trim()
    .notEmpty()
    .withMessage("Please enter your last name.")
    .isAlpha("en-US", { ignore: " " }) // Allow alphabets and spaces
    .withMessage("Last name can only contain alphabets and spaces."),

  catchAsync(async (req, res, next) => {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({
        status: false,
        error: errors.array()[0].msg,
      });
    }

    const { firstName, lastName } = req.body;

    try {
      // Generate a random email string
      const randomEmail =
        crypto.randomBytes(6).toString("hex") + "@example.com";

      // Create the TempUser with the random email
      const tempUser = await TempUser.create({
        firstName,
        lastName,
        email: randomEmail, // Set random email
      });

      res.status(200).json({
        status: true,
        message: "Proceed to the next step: Email",
        tempUserId: tempUser._id,
      });
    } catch (error) {
      console.error(error);
      res.status(500).json({
        status: false,
        error: "Something went wrong while processing name.",
      });
    }
  }),
];


exports.signupEmail = [
  body("email")
    .isEmail()
    .withMessage("Please provide a valid email address.")
    .normalizeEmail(),

  catchAsync(async (req, res, next) => {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({
        status: false,
        error: errors.array()[0].msg,
      });
    }

    const { email } = req.body;
    const { tempUserId } = req.params;

    try {
      // Check if the TempUser exists
      const tempUser = await TempUser.findById(tempUserId);
      if (!tempUser) {
        return res.status(404).json({
          status: false,
          error: "No user found with this ID.",
        });
      }

      // Check if the email is already registered
      const user = await User.findOne({ email });
      if (user) {
        return res.status(400).json({
          status: false,
          error: "Email is already registered.",
        });
      }

      // Update the TempUser with the provided email
      await TempUser.findByIdAndUpdate(tempUserId, { email });

      res.status(200).json({
        status: true,
        message: "Proceed to the next step: Password",
      });
    } catch (error) {
      console.error(error);
      res.status(500).json({
        status: false,
        error: "Something went wrong while processing email.",
      });
    }
  }),
];

exports.signupPassword = [
  body("password")
    .notEmpty()
    .withMessage("Please enter a password.")
    .isLength({ min: 8 })
    .withMessage("Password must be at least 8 characters long.")
    .matches(/[0-9]/)
    .withMessage("Password must contain at least one number.")
    .matches(/[A-Z]/)
    .withMessage("Password must contain at least one uppercase letter.")
    .matches(/[a-z]/)
    .withMessage("Password must contain at least one lowercase letter."),

  catchAsync(async (req, res, next) => {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({
        status: false,
        error: errors.array()[0].msg,
      });
    }

    const { password } = req.body;
    const { tempUserId } = req.params;
    // console.log(password);

    try {
      // Check if the TempUser exists
      const tempUser = await TempUser.findById(tempUserId);
      if (!tempUser) {
        return res.status(404).json({
          status: false,
          error: "No user found with this ID.",
        });
      }

      // Hash the password before saving it
      // const hashedPassword = await bcrypt.hash(password, 12);

      // Update the TempUser with the hashed password
      await TempUser.findByIdAndUpdate(tempUserId, {
        password: password,
      });

      res.status(200).json({
        status: true,
        message:
          "Password saved successfully. Proceed to the next step: Profile Info",
      });
    } catch (error) {
      console.error(error);
      res.status(500).json({
        status: false,
        message: "Something went wrong while processing password.",
      });
    }
  }),
];



exports.signupProfileInfo = catchAsync(async (req, res, next) => {
  const { location, recentJob, company, longitude, latitude } = req.body;
  const { tempUserId } = req.params;

  try {
    // ✅ Check if required fields are provided
    if (!recentJob || !company) {
      return res.status(400).json({
        status: false,
        error: "Please provide all required fields (recentJob, company).",
      });
    }

    let placeId = null;
    let locationName = location || "Unknown Location";
    let isValidPlace = false;

    // ✅ If coordinates are provided, resolve place details via helper
    if (latitude && longitude) {
      const details = await getPlaceDetails(latitude, longitude, location);

      placeId = details.placeId;
      locationName = details.locationName;
      isValidPlace = details.isValidPlace;
    }

    // ✅ Check if the TempUser exists
    const tempUser = await TempUser.findById(tempUserId);
    if (!tempUser) {
      return res.status(404).json({
        status: false,
        error: "No user found with this ID.",
      });
    }

    // ✅ Update TempUser profile information (allowing nullable fields)
    await TempUser.findByIdAndUpdate(tempUserId, {
      longitude: longitude || null,
      latitude: latitude || null,
      placeId: placeId || null,
      location: locationName || null,
      recentJob,
      company,
      isValidPlace,
    });

    // ✅ Send response
    res.status(200).json({
      status: true,
      message:
        "Profile info saved successfully. Proceed to the next step: Interests",
      data: {
        tempUserId,
        longitude: longitude || null,
        latitude: latitude || null,
        placeId: placeId || null,
        location: locationName || null,
        recentJob,
        company,
        isValidPlace,
      },
    });
  } catch (error) {
    console.error("❌ Error processing profile info:", error.message);
    res.status(500).json({
      status: false,
      error: "Something went wrong while processing profile info.",
    });
  }
});



exports.signupInterests = catchAsync(async (req, res, next) => {
  const { interests } = req.body;
  const { tempUserId } = req.params;

  try {
    // Check if interests are provided and within the allowed range (3 to 5)
    if (!interests || !Array.isArray(interests) || interests.length < 3 || interests.length > 5) {
      return res.status(400).json({
        status: false,
        error: "Please select between 3 to 5 interests.",
      });
    }

    // Check if the TempUser exists
    const tempUser = await TempUser.findById(tempUserId);
    if (!tempUser) {
      return res.status(404).json({
        status: false,
        error: "No user found with this ID.",
      });
    }

    // Update TempUser interests
    await TempUser.findByIdAndUpdate(tempUserId, { interests });

    res.status(200).json({
      status: true,
      message: "Interests saved successfully. Proceed to the next step: Upload Photo.",
    });
  } catch (error) {
    console.error("Error saving interests:", error);
    res.status(500).json({
      status: false,
      error: "Something went wrong while saving interests.",
    });
  }
});

exports.signupPhoto = [
  upload.single("photo"),
  async (req, res) => {
    const { tempUserId } = req.params;

    try {
      // Validate if TempUser exists
      const tempUser = await TempUser.findById(tempUserId);
      if (!tempUser) {
        return res.status(404).json({
          status: false,
          error: "No user found with this ID.",
        });
      }

      // Handle photo upload
      if (req.file) {
        tempUser.photo = req.file.path; // Update the photo path
        console.log("Photo uploaded:", req.file.path);
      }

      // Generate and add OTP
      const otp = generateOTP();
      tempUser.otps.push({
        otp,
        otpExpires: Date.now() + 10 * 60 * 1000, // OTP valid for 10 minutes
      });

      // Save the updated user data
      await tempUser.save();

      // Generate URL for user actions (e.g., activation or profile setup)
      // const url = `${req.protocol}://${req.get("host")}/me`;

      // Send OTP email
      await new Email(tempUser, { OTP: otp }).sendPasswordReset();
      console.log("OTP email sent successfully!");

      // Respond with success
      res.status(200).json({
        status: true,
        message: "Photo uploaded and OTP sent successfully.",
        otp,
      });
    } catch (error) {
      console.error("Error in signupPhoto:", error);

      // Respond with error details
      res.status(500).json({
        status: false,
        error: error,
        // error: "An error occurred while processing your request.",
      });
    }
  },
];

exports.resendOTP = async (req, res) => {
  try {
    const { tempUserId } = req.params;

    // Fetch the TempUser from the database
    const tempUser = await TempUser.findById(tempUserId);

    if (!tempUser) {
      return res.status(404).json({ status: false, error: "User not found." });
    }

    // Generate a new OTP
    const otp = generateOTP();

    // Update OTP in the TempUser's otps array
    tempUser.otps.push({
      otp: otp,
      otpExpires: Date.now() + 10 * 60 * 1000, // OTP valid for 10 minutes
    });

    // Save the updated TempUser document
    await tempUser.save();

    // Send the OTP via email
    await new Email(tempUser, { OTP: otp }).sendPasswordReset();
    console.log("OTP email sent!");

    // Return success response
    res.status(200).json({
      status: true,
      message: "A new OTP has been sent to your email.",
    });
  } catch (error) {
    console.error("Error while resending OTP:", error);
    res.status(500).json({
      status: false,
      error: "Something went wrong while resending the OTP.",
    });
  }
};
exports.verifyOTP = async (req, res, next) => {
  const { otp, fcmToken } = req.body;   // <-- added fcmToken
  const { tempUserId } = req.params;

  try {
    // Fetch the TempUser from the database
    const tempUser = await TempUser.findById(tempUserId);

    if (!tempUser) {
      return res.status(404).json({ status: false, error: "User not found." });
    }

    // Check for a valid OTP in the otps array
    const validOtp = tempUser.otps.find(
      (otpEntry) =>
        otpEntry.otp === otp &&
        Date.now() < new Date(otpEntry.otpExpires).getTime()
    );

    if (!validOtp) {
      return res.status(400).json({
        status: false,
        error: "Invalid or expired OTP.",
      });
    }

    // Check if password is set for the user
    if (!tempUser.password) {
      return res.status(400).json({
        status: false,
        error: "Password is missing in the user data.",
      });
    }

    // If OTP is valid, create the final user in the User collection
    const user = await User.create({
      firstName: tempUser.firstName,
      lastName: tempUser.lastName,
      email: tempUser.email,
      password: tempUser.password,
      placeId: tempUser.placeId,
      location: tempUser.location,
      recentJob: tempUser.recentJob,
      company: tempUser.company,
      latitude: tempUser.latitude,
      longitude: tempUser.longitude,
      interests: tempUser.interests,
      photo: tempUser.photo || null,
      fcmToken: fcmToken || null,   // <-- save the FCM token if provided
    });

    // Generate a URL for the user (e.g., a profile URL or activation URL)
    const url = `${req.protocol}://${req.get("host")}/me`;

    // Send the welcome email
    await new Email(tempUser).sendWelcome();

    // Delete the temporary user document after successful verification
    await TempUser.findByIdAndDelete(tempUserId);

    // Send the created user a JWT token
    createSendToken(user, 201, req, res);
  } catch (error) {
    console.error("Error while verifying OTP:", error);
    res.status(500).json({
      status: false,
      error: "Error occurred while verifying OTP.",
    });
  }
};


exports.login = catchAsync(async (req, res, next) => {
  const { email, password, fcmToken } = req.body; // <-- accept fcmToken from client

  // 1) Check if email and password exist
  if (!email || !password) {
    return res.status(400).json({
      status: false,
      error: "Please provide email and password!",
    });
  }

  // 2) Check if user exists && password is correct
  const user = await User.findOne({ email }).select("+password");

  if (!user || !(await user.correctPassword(password, user.password))) {
    return res.status(401).json({
      status: false,
      error: "Incorrect email or password",
    });
  }

  // 3) If fcmToken is provided, update it in the user record
  if (fcmToken) {
    user.fcmToken = fcmToken;
    await user.save({ validateBeforeSave: false }); // skip validation for speed
    console.log(`📱 Updated FCM token for user ${user._id}`);
  }

  // 4) Send token to client
  createSendToken(user, 200, req, res);
});


// Protect middleware
exports.protect = async (req, res, next) => {
  try {
    let token;

    // 1) Get token from Authorization header or cookies
    if (
      req.headers.authorization &&
      req.headers.authorization.startsWith("Bearer")
    ) {
      token = req.headers.authorization.split(" ")[1];
    } else if (req.cookies.jwt) {
      token = req.cookies.jwt;
    }

    // 2) If no token
    if (!token) {
      return res.status(401).json({
        status: false,
        message: "You are not logged in! Please log in to get access.",
      });
    }

    // 3) Verify token
    const decoded = jwt.verify(token, process.env.JWT_SECRET);

    // 4) Check if user still exists
    const currentUser = await User.findById(decoded.id);
    if (!currentUser) {
      return res.status(401).json({
        status: false,
        message: "The user belonging to this token no longer exists.",
      });
    }

    // 5) Enforce single-device login
    if (currentUser.activeToken !== token) {
      return res.status(401).json({
        status: false,
        message: "Session expired. You logged in from another device.",
      });
    }

    // 6) Grant access
    req.user = currentUser;
    next();
  } catch (error) {
    console.error("Error in protect middleware:", error);

    if (error.name === "JsonWebTokenError") {
      return res.status(401).json({
        status: false,
        message: "Invalid token. Please log in again.",
      });
    }

    if (error.name === "TokenExpiredError") {
      return res.status(401).json({
        status: false,
        message: "Token expired. Please log in again.",
      });
    }

    return res.status(401).json({
      status: false,
      message: "Authorization failed.",
    });
  }
};
// Only for rendered pages, no errors!
exports.isLoggedIn = async (req, res, next) => {
  if (req.cookies.jwt) {
    try {
      // 1) verify token
      const decoded = await promisify(jwt.verify)(
        req.cookies.jwt,
        process.env.JWT_SECRET
      );

      // 2) Check if user still exists
      const currentUser = await User.findById(decoded.id);
      if (!currentUser) {
        return next();
      }

      // 3) Check if user changed password after the token was issued
      if (currentUser.changedPasswordAfter(decoded.iat)) {
        return next();
      }

      // THERE IS A LOGGED IN USER
      res.locals.user = currentUser;
      return next();
    } catch (err) {
      return next();
    }
  }
  next();
};

exports.restrictTo = (...roles) => {
  return (req, res, next) => {
    // roles ['admin', 'lead-guide']. role='user'
    if (!roles.includes(req.user.role)) {
      return next(
        new AppError("You do not have permission to perform this action", 403)
      );
    }

    next();
  };
};

exports.updatePassword = [
  // Validate current password, new password, and confirm password
  body("passwordCurrent")
    .notEmpty()
    .withMessage("Please enter your current password."),

  body("password")
    .notEmpty()
    .withMessage("Please enter a new password.")
    .isLength({ min: 8 })
    .withMessage("Password must be at least 8 characters long.")
    .matches(/[0-9]/)
    .withMessage("Password must contain at least one number.")
    .matches(/[A-Z]/)
    .withMessage("Password must contain at least one uppercase letter.")
    .matches(/[a-z]/)
    .withMessage("Password must contain at least one lowercase letter.")
    .trim(), // Trim whitespace around the password

  body("passwordConfirm")
    .notEmpty()
    .withMessage("Please confirm your new password.")
    .custom((value, { req }) => value === req.body.password)
    .withMessage("Passwords do not match."),

  // Actual password update logic
  catchAsync(async (req, res, next) => {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({
        status: false,
        errors: errors.array()[0].msg, // Return only the first validation error
      });
    }

    // 1) Get user from the database, including the password field
    const user = await User.findById(req.user.id).select("+password");
    if (!user) {
      return res.status(404).json({
        status: false,
        error: "User not found.",
      });
    }

    // 2) Check if the provided current password is correct
    const isPasswordCorrect = await user.correctPassword(
      req.body.passwordCurrent,
      user.password
    );
    if (!isPasswordCorrect) {
      return res.status(401).json({
        status: false,
        error: "Your current password is incorrect.",
      });
    }

    // 3) If the current password is correct, update the password
    user.password = req.body.password;
    user.passwordConfirm = req.body.passwordConfirm;
    await user.save();

    // 4) Log the user in and send JWT token
    createSendToken(user, 200, req, res); // Function to create and send the JWT
  }),
];
exports.logout = catchAsync(async (req, res, next) => {
  try {
    const userId = req.user._id; // Get the authenticated user's ID
 try {
          await User.findByIdAndUpdate(userId, { fcmToken: null });
        } catch (dbError) {
          return next(new AppError("Error updating user FCM token", 500)); // Handle database update errors
        }
    // Log the user out
    req.logout(async (err) => {
      if (err) {
        return next(new AppError("Error logging out", 500)); // Handle logout errors
      }

      // Clear the session
      req.session.destroy(async (sessionErr) => {
        if (sessionErr) {
          return next(new AppError("Error clearing session", 500)); // Handle session clear errors
        }

        // Set the user's FCM token to null in the database
       

        // Set the JWT cookie to 'loggedout' with a short expiry
        res.cookie("jwt", "loggedout", {
          expires: new Date(Date.now() + 10 * 1000), // Cookie expires in 10 seconds
          httpOnly: true, // Ensures cookie is not accessible via JavaScript
          secure: process.env.NODE_ENV === "production", // Ensure the cookie is only sent over HTTPS in production
          sameSite: "strict", // Ensures the cookie is sent only for same-site requests
        });

        // Send success response
        res.status(200).json({
          status: true,
          message: "Logged out successfully",
        });
      });
    });
  } catch (err) {
    return next(new AppError("Something went wrong during logout", 500));
  }
});
exports.checkAuth = (req, res) => {
  try {
    res.status(200).json(req.user);
  } catch (error) {
    console.log("Error in checkAuth controller", error.message);
    res.status(500).json({ message: "Internal Server Error" });
  }
};
exports.deleteAccount = async (req, res) => {
  try {
    const userId = req.user._id; // Logged-in user ID

    // Fetch the user
    const user = await User.findById(userId);
    if (!user) {
      return res.status(404).json({ status: false, message: "User not found." });
    }

    // Step 1: Delete the user's messages
    await Message.deleteMany({ $or: [{ senderId: userId }, { receiverId: userId }] });

    // Step 2: Delete the user's connections
    await Connection.deleteMany({ $or: [{ sender: userId }, { receiver: userId }] });

    // Step 3: Delete the user's account
    await User.findByIdAndDelete(userId);

    // Step 4: Log the user out (invalidate the token)
    res.clearCookie("jwt"); // Clear the JWT cookie (if using cookies)
    res.status(200).json({
      status: true,
      message: "Account deleted successfully.",
    });
  } catch (error) {
    console.error("Error deleting account:", error);
    res.status(500).json({ status: false, message: "Internal server error" });
  }
};

exports.updateProfilePicture = catchAsync(async (req, res, next) => {
  // Handle the upload process
  upload.single("photo")(req, res, async (err) => {
    try {
      if (err) {
        return res.status(400).json({
          status: false,
          error: err.message || "An error occurred while uploading the photo.",
        });
      }

      // Validate user ID
      const { id } = req.params;
      if (!mongoose.Types.ObjectId.isValid(id)) {
        return res.status(400).json({ status: false, error: "Invalid user ID format." });
      }

      // Find user
      const user = await User.findById(id);
      if (!user) {
        return res.status(404).json({ status: false, error: "User not found." });
      }

      // If no file is uploaded, return an error
      if (req.file) {
         // Normalize file path
      let photoPath = req.file.path.replace(/\\/g, "/");

      // Update user profile picture
      user.photo = photoPath;
      await user.save();
      }

     
     

      // Send token with updated user data
      createSendToken(user, 200, req, res);
    } catch (error) {
      console.error("Profile picture update error:", error);
      res.status(500).json({ status: false, error: "An error occurred while updating the profile picture." });
    }
  });
});
