const appleSignin = require("apple-signin-auth");
const jwt = require("jsonwebtoken");
const User = require("../models/userModel"); // your user model

// 🔑 Centralized helper inside controller
const createSendToken = async (user, statusCode, req, res) => {
  // 1) Generate JWT
  const token = jwt.sign({ id: user._id }, process.env.JWT_SECRET, {
    expiresIn: "7d",
  });

  // 2) Save activeToken in DB (enforce single-device login)
  user.activeToken = token;
  await user.save({ validateBeforeSave: false });

  // 3) Cookie config (for web clients)
  const cookieExpireInDays = parseInt(process.env.JWT_COOKIE_EXPIRE_IN, 10) || 90;
  const cookieOptions = {
    expires: new Date(Date.now() + cookieExpireInDays * 24 * 60 * 60 * 1000),
    httpOnly: true,
    secure: req.secure || req.headers["x-forwarded-proto"] === "https",
    sameSite: "strict",
  };
  res.cookie("jwt", token, cookieOptions);

  // 4) Hide sensitive fields
  user.password = undefined;

  // 5) Send response
  res.status(statusCode).json({
    status: true,
    token,
    data: {
      user: {
        _id: user._id,
        firstName: user.firstName || "",
        lastName: user.lastName || "",
        email: user.email,
        role: user.role,
        photo: user.photo || null,
        interests: user.interests || [],
        fcmToken: user.fcmToken || null,
      },
    },
  });
};

exports.appleLogin = async (req, res) => {
  const { id_token, firstName, lastName } = req.body;

  if (!id_token) {
    return res.status(400).json({
      status: false,
      message: "Missing id_token in request body",
    });
  }

  console.log("Verifying Apple ID token with client ID:", process.env.APPLE_CLIENT_ID);

  try {
    // 1) Verify token from Apple
    const appleData = await appleSignin.verifyIdToken(id_token, {
      audience: [process.env.APPLE_CLIENT_ID],
      ignoreExpiration: true,
    });

    console.log("Apple token verified successfully:", appleData);

    const { sub: appleId, email } = appleData;

    // 2) Check if user exists, else create new one
    let user = await User.findOne({ email });
    if (!user) {
      user = await User.create({
        firstName: firstName || "AppleUser", // Apple shares real name only first time
        lastName: lastName || "",
        email,
        appleId,
        password: "apple_placeholder", // not used for login
      });
    }

    // 3) Issue JWT and respond using centralized helper
    return createSendToken(user, 200, req, res);
  } catch (err) {
    console.error("Apple login error:", err);
    res.status(400).json({ status: false, message: err.message });
  }
};
