import Message from "../../models/messageModel.js";
import User from "../../models/userModel.js";
import mongoose from "mongoose";

export let messageController = {
  async getAllConversations(req, res) {
    const page = parseInt(req.query.page) || 1;
    const limit = parseInt(req.query.limit) || 10;
    const skip = (page - 1) * limit;
    const search = req.query.search || "";
    const status = req.query.status;
    const venue = req.query.venue;

    try {
      const pipeline = [
        ...(search
          ? [
              {
                $match: {
                  $or: [
                    {
                      firstName: {
                        $regex: search,
                        $options: "i",
                      },
                    },
                    {
                      lastName: {
                        $regex: search,
                        $options: "i",
                      },
                    },
                    {
                      email: {
                        $regex: search,
                        $options: "i",
                      },
                    },
                  ],
                },
              },
            ]
          : []),
        ...(status
          ? [{ $match: { status: { $regex: status, $options: "i" } } }]
          : []),
        ...(venue
          ? [{ $match: { location: { $regex: venue, $options: "i" } } }]
          : []),

        {
          $lookup: {
            from: "messages",
            let: { userId: "$_id" },
            pipeline: [
              {
                $match: {
                  $expr: {
                    $or: [
                      { $eq: ["$senderId", "$$userId"] },
                      {
                        $eq: ["$receiverId", "$$userId"],
                      },
                    ],
                  },
                },
              },
              {
                $addFields: {
                  partnerId: {
                    $cond: [
                      { $eq: ["$senderId", "$$userId"] },
                      "$receiverId",
                      "$senderId",
                    ],
                  },
                },
              },
              {
                $group: {
                  _id: "$partnerId",
                },
              },
            ],
            as: "partners",
          },
        },
        {
          $addFields: {
            totalConversations: { $size: "$partners" },
          },
        },
        {
          $project: {
            _id: 1,
            firstName: 1,
            lastName: 1,
            email: 1,
            photo: 1,
            location: 1,
            totalConversations: 1,
          },
        },
        { $sort: { createdAt: -1 } },
        { $skip: skip },
        { $limit: limit },
      ];

      const users = await User.aggregate(pipeline);

      const countQuery = {
        ...(search && {
          $or: [
            { firstName: { $regex: search, $options: "i" } },
            { lastName: { $regex: search, $options: "i" } },
            { email: { $regex: search, $options: "i" } },
          ],
        }),
        ...(status && { status: { $regex: status, $options: "i" } }),
        ...(venue && { location: { $regex: venue, $options: "i" } }),
      };

      const total = await User.countDocuments(countQuery);

      return res.json({
        status: true,
        data: users,
        total,
        page,
        limit,
      });
    } catch (error) {
      console.error("getAllConversations error:", error);
      return res.status(500).json({ status: "error", message: "Server error" });
    }
  },

  async getUserConversations(req, res) {
    const userId = req.params.id;

    try {
      const conversations = await Message.aggregate([
        {
          $match: {
            $or: [
              { senderId: new mongoose.Types.ObjectId(userId) },
              { receiverId: new mongoose.Types.ObjectId(userId) },
            ],
          },
        },
        {
          $addFields: {
            partnerId: {
              $cond: [
                { $eq: ["$senderId", new mongoose.Types.ObjectId(userId)] },
                "$receiverId",
                "$senderId",
              ],
            },
          },
        },
        { $sort: { createdAt: -1 } },
        {
          $group: {
            _id: "$partnerId",
            lastMessage: { $first: "$$ROOT" },
            totalMessages: { $sum: 1 },
          },
        },
        {
          $lookup: {
            from: "users",
            localField: "_id",
            foreignField: "_id",
            as: "partner",
          },
        },
        {
          $unwind: "$partner",
        },
        {
          $project: {
            partnerId: "$_id",
            totalMessages: 1,
            lastMessage: {
              text: "$lastMessage.text",
              createdAt: "$lastMessage.createdAt",
              seen: "$lastMessage.seen",
              delivered: "$lastMessage.delivered",
            },
            partner: {
              _id: 1,
              firstName: 1,
              lastName: 1,
              email: 1,
              photo: 1,
            },
          },
        },
        { $sort: { "lastMessage.createdAt": -1 } },
      ]);

      return res.json({
        status: true,
        data: conversations,
      });
    } catch (error) {
      console.error("getUserConversations error:", error);
      return res.status(500).json({ status: "error", message: "Server error" });
    }
  },
};
