import Connection from "../../models/connectionModel.js";

export let connectionController = {
    async getAllConnections(req, res) {
        try {
            const page = parseInt(req.query.page) || 1;
            const limit = parseInt(req.query.limit) || 10;
            const search = req.query.search || "";
            const status = req.query.status || "";
            const venueType = req.query.venueType || "";
            const skip = (page - 1) * limit;

            const query = {};

            if (search) {
                query.$or = [
                    { "sender.firstName": { $regex: search, $options: "i" } },
                    { "sender.lastName": { $regex: search, $options: "i" } },
                    { "receiver.firstName": { $regex: search, $options: "i" } },
                    { "receiver.lastName": { $regex: search, $options: "i" } },
                ];
            }

            if (status && status !== "all") {
                const statusMap = {
                    pending: "pending",
                    accepted: "accepted",
                    met: "accepted",
                    blocked: "rejected",
                };
                query.status = statusMap[status] || status;
            }

            if (venueType && venueType !== "all") {
                query.venueType = { $regex: venueType, $options: "i" };
            }

            const connections = await Connection.find(query)
                .populate([
                    {
                        path: "sender",
                        select: "_id firstName lastName email photo location",
                    },
                    {
                        path: "receiver",
                        select: "_id firstName lastName email photo location",
                    },
                ])
                .sort({ createdAt: -1 })
                .skip(skip)
                .limit(limit)
                .lean();

            const totalConnections = await Connection.countDocuments(query);
            const successfulConnections = await Connection.countDocuments({
                ...query,
                status: "accepted",
            });
            const pendingConnections = await Connection.countDocuments({
                ...query,
                status: "pending",
            });
            const successRatio = totalConnections
                ? Math.round((successfulConnections / totalConnections) * 100)
                : 0;

            const formattedConnections = connections.map((conn) => ({
                _id: conn._id,
                fromUser: `${conn.sender?.firstName || "Unknown"} ${
                    conn.sender?.lastName || ""
                }`,
                toUser: `${conn.receiver?.firstName || "Unknown"} ${
                    conn.receiver?.lastName || ""
                }`,
                date: new Date(conn.createdAt).toLocaleDateString(),
                venueType: "Unknown",
                venueName: conn.sender?.location || "Unknown",
                status:
                    conn.status === "rejected"
                        ? "Blocked"
                        : conn.status === "accepted"
                        ? "Accepted"
                        : conn.status === "pending"
                        ? "Pending"
                        : "Pending",
                connectionTime: null,
            }));

            return res.json({
                status: "success",
                data: {
                    connections: formattedConnections,
                    totalConnections,
                    successfulConnections,
                    pendingConnections,
                    successRatio,
                    page,
                    limit,
                    totalPages: Math.ceil(totalConnections / limit),
                },
            });
        } catch (error) {
            console.error("getAllConnections error:", error);
            return res
                .status(500)
                .json({ status: "error", message: "Server error" });
        }
    },
};
